package org.mulesoft.als.suggestions.plugins.jsonschema.agentnetwork

import amf.core.client.scala.model.domain.{AmfObject, AmfScalar}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.als.suggestions.RawSuggestion
import org.mulesoft.als.suggestions.aml.AmlCompletionRequest
import org.mulesoft.als.suggestions.jsonschema.declarations.InferredDeclarationProvider.DeclarationType
import org.mulesoft.als.suggestions.plugins.jsonschema.JsonSchemaBaseCompletionPlugin
import org.mulesoft.amfintegration.amfconfiguration.executioncontext.Implicits.global
import org.mulesoft.amfintegration.vocabularies.jsonschema.agentnetwork.AgentNetworkVocabulary

import scala.concurrent.Future

trait BaseProviderReferencesCompletionPlugin extends JsonSchemaBaseCompletionPlugin {
  protected val referenceIri: DeclarationType = AgentNetworkVocabulary.REFERENCE

  override def innerResolve(request: AmlCompletionRequest): Future[Seq[RawSuggestion]] =
    if(!applies(request)) emptySuggestion
    else
      Future {
        getDeclarationKind(request) match {
          case Some(declarationType) =>
            getSuggestionsForDeclaration(request, declarationType)
          case _ =>
            Seq.empty
        }
      }

  protected def getDeclarationKind(request: AmlCompletionRequest): Option[DeclarationType] =
    AgentNetworkVocabulary.getKind(request.amfObject)

  private def getSuggestionsForDeclaration(request: AmlCompletionRequest, declarationType: DeclarationType) =
    request.inferredDeclarationProvider.getDeclarationFor(declarationType)
      .collect {
        case element: JsonLDObject => element
      }
      .flatMap(getName)
      .map(toRaw)

  protected def toRaw(name: String): RawSuggestion =
    RawSuggestion.plain(name, "reference")

  private def getName(element: JsonLDObject) =
    element.fields.fields().map(_.field.value.name)

  protected def applies(request: AmlCompletionRequest): Boolean =
    isInsideReference(request)  && isInsideNameFacet(request)

  private def isInsideNameFacet(request: AmlCompletionRequest) =
    request.fieldEntry.map(_.field.value.iri()).contains(AgentNetworkVocabulary.NAME)

  private def isInsideReference(request: AmlCompletionRequest) = request.amfObject match {
      case json: JsonLDObject => json.typeIris.contains(referenceIri)
      case _ => false
    }
}

object LocalBaseProviderReferencesCompletionPlugin extends BaseProviderReferencesCompletionPlugin {
  override def id: String = "LocalBaseProviderReferencesCompletionPlugin"
}

object LocalConnectionKindProviderReferencesCompletionPlugin extends BaseProviderReferencesCompletionPlugin {
  override def id: String = "LocalConnectionKindProviderReferencesCompletionPlugin"

  override protected val referenceIri: DeclarationType = AgentNetworkVocabulary.REF

  override protected def applies(request: AmlCompletionRequest): Boolean = request.prefix.isEmpty && super.applies(request)

  override protected def getDeclarationKind(request: AmlCompletionRequest): Option[DeclarationType] = {
    request.branchStack.collectFirst{
        case obj: AmfObject if obj.fields.fields().exists(_.field.value.iri() == AgentNetworkVocabulary.KIND) =>
          obj.fields.fields().find(_.field.value.iri() == AgentNetworkVocabulary.KIND)
      }.flatten
      .map(_.value.value)
      .collect{
        case AmfScalar(value, _) => value.toString
      }.map(AgentNetworkVocabulary.getKind)
  }
}
