package amf.xml.internal

import amf.core.client.common.remote.Content
import amf.core.client.common.validation.SeverityLevels.VIOLATION
import amf.core.client.common.validation.{ProfileNames, ValidationMode}
import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.model.document.PayloadFragment
import amf.core.client.scala.model.domain.{ScalarNode, Shape}
import amf.core.client.scala.validation.payload.{AMFShapePayloadValidator, ShapeValidationConfiguration}
import amf.core.client.scala.validation.{AMFValidationReport, AMFValidationResult}
import amf.shapes.client.scala.model.domain.AnyShape
import amf.xml.internal.transformer.TypeToXmlSchema
import amf.xml.internal.transformer.TypeToXmlSchema.getTargetNamespace
import amf.xml.internal.util.ClassLoadingUtils.withContextClassLoader
import amf.xml.internal.util.XmlUtils
import amf.xml.internal.util.XmlUtils.{DisallowDoctypeDeclFeature, ExpandEntities, isXMLValue}
import org.xml.sax.helpers.XMLFilterImpl
import org.xml.sax.{Attributes, InputSource, SAXException, XMLReader}

import java.io.{ByteArrayInputStream, StringReader}
import java.net.URI
import java.nio.charset.StandardCharsets.UTF_8
import javax.xml.transform.sax.SAXSource
import javax.xml.transform.stream.StreamSource
import javax.xml.validation.SchemaFactory
import scala.concurrent.duration.DurationInt
import scala.concurrent.{Await, ExecutionContext, Future}

case class XmlPayloadValidator(shape: Shape, config: ShapeValidationConfiguration, validationMode: ValidationMode) extends AMFShapePayloadValidator {
  private val XmlTargetProperty = Option("http://www.w3.org/2001/XMLSchema#")
  private val ValidationId = "http://www.w3.org/2001/XMLSchema#payload-validation-error"

  lazy val xsd: String = shape match {
    case s: AnyShape => TypeToXmlSchema.getSchemaAsString(s)
    case _ => ""
  }


  override def validate(payload: String): Future[AMFValidationReport] = {
    implicit val executionContext: ExecutionContext = ExecutionContext.Implicits.global
    Future {
      executeValidation(shape, payload, None)
    }
  }

  override def validate(payloadFragment: PayloadFragment): Future[AMFValidationReport] = {
    implicit val executionContext: ExecutionContext = ExecutionContext.Implicits.global
    Future {
      val dataNode = payloadFragment.encodes
      dataNode match {
        case scalar: ScalarNode => executeValidation(shape, scalar.value.value(), Some(scalar))
        case _ => throw new RuntimeException("Unsupported Operation")
      }
    }
  }

  override def syncValidate(payload: String): AMFValidationReport = executeValidation(shape, payload, None)

  private def executeValidation(shape: Shape, payload: String, node: Option[ScalarNode]): AMFValidationReport = {
    val validationResult = try {
      if (shape.getClass == classOf[AnyShape]) {
        // Check if at least payload is xml like
        if (isXMLValue(payload)) None
        else {
          Some(buildValidationResult("Payload is not an XML", node))
        }
      } else shape match {
        case anyShape: AnyShape => //It means extending AnyShape
          // Validate against xml schema
          val targetNamespace = getTargetNamespace(shape)
          validateAgainstSchema(payload, xsd, targetNamespace, anyShape.location.orNull)
          None
        case _ => throw new RuntimeException("Unsupported Operation")
      }
    } catch {
      case e: Exception =>
        Some(buildValidationResult(e.getMessage, node))
    }
    AMFValidationReport("", // todo some url like ""http://test.com/xml-payload"
      ProfileNames.PAYLOAD,
      validationResult.toList,
    )
  }

  private def buildValidationResult(message: String, exampleNode: Option[ScalarNode]): AMFValidationResult =
    AMFValidationResult(message,
      VIOLATION,
      exampleNode.map(_.id).getOrElse(""),
      XmlTargetProperty,
      ValidationId,
      exampleNode.flatMap(_.position()),
      exampleNode.flatMap(_.location()),
      this)

  @throws[Exception]
  private def validateAgainstSchema(value: String, xsd: String, targetNamespace: Option[String], location: String): Unit = {

    val factory = XmlSchemaFactory(location, AmfConfigContentFetcher(config))
    val schema = factory.newSchema(new StreamSource(new StringReader(xsd), location))
    val validator = schema.newValidator

    val xmlReader: XMLReader = XmlUtils.createXMLReader

    val stream = new ByteArrayInputStream(value.getBytes(UTF_8))
    validator.validate(new SAXSource(NamespaceFilter(xmlReader, targetNamespace), new InputSource(stream)))
  }

  case class NamespaceFilter(parent: XMLReader, requiredNamespace: Option[String]) extends XMLFilterImpl(parent) {
    @throws[SAXException]
    override def startElement(arg0: String, arg1: String, arg2: String, arg3: Attributes): Unit = {
      requiredNamespace match {
        case Some(rn) if arg0 != rn => super.startElement(rn, arg1, arg2, arg3)
        case _ => super.startElement(arg0, arg1, arg2, arg3)
      }

    }
  }
}

private[xml] case class AmfConfigContentFetcher(private val config: ShapeValidationConfiguration) extends ContentFetcher {
  override def fetchContent(url: String): Future[Content] = config.fetchContent(url)
}
