package amf.xml.internal

import amf.core.client.common.remote.Content
import amf.xml.internal.util.ClassLoadingUtils.withContextClassLoader
import amf.xml.internal.util.XmlUtils.{DisallowDoctypeDeclFeature, ExpandEntities}
import org.w3c.dom.bootstrap.DOMImplementationRegistry
import org.w3c.dom.ls.DOMImplementationLS

import java.io.ByteArrayInputStream
import java.net.URI
import javax.xml.validation.SchemaFactory
import scala.concurrent.duration.DurationInt
import scala.concurrent.{Await, Future}

trait ContentFetcher {
  def fetchContent(location: String): Future[Content]
}

object XmlSchemaFactory {

  def apply(location: String, fetcher: ContentFetcher): SchemaFactory = {
    val registry = DOMImplementationRegistry.newInstance
    val domImplementationLS = registry.getDOMImplementation("LS").asInstanceOf[DOMImplementationLS]

    System.setProperty(
      "javax.xml.validation.SchemaFactory:http://www.w3.org/XML/XMLSchema/v1.1",
      "org.apache.xerces.jaxp.validation.XMLSchema11Factory"
    )

    val factory = withContextClassLoader(this.getClass.getClassLoader, () => SchemaFactory.newInstance("http://www.w3.org/XML/XMLSchema/v1.1"))
    factory.setFeature(DisallowDoctypeDeclFeature, !ExpandEntities)
    factory.setResourceResolver((`type`: String, namespaceURI: String, publicId: String, systemId: String, baseURI: String) => {
      val importLocation =
        if (new URI(systemId).getScheme != null) systemId
        else {
          val rootURI = if (systemId.startsWith("/")) location else baseURI

          if (rootURI.endsWith("/")) rootURI + systemId
          else rootURI.substring(0, rootURI.lastIndexOf("/") + 1) + systemId
        }


      val eventualContent = fetcher.fetchContent(importLocation)
      val content = Await.result(eventualContent, 5.seconds)
      val input = domImplementationLS.createLSInput
      input.setByteStream(new ByteArrayInputStream(content.stream.toString.getBytes))
      input.setSystemId(systemId)
      input.setBaseURI(baseURI)
      input
    })
    factory
  }
}
