package ang.umi.commands

import org.mulesoft.common.io.{Fs, SyncFile}
import scopt.OptionParser

import scala.util.Random

case class ParserConfig(angURL: Option[String] = None,
                        output: Option[SyncFile] = None,
                        instances: Int = 1000,
                        cardinality: Int = 3,
                        parallelism: Int = 10,
                        seed: Long = Random.nextLong(),
                        clientID: Option[String] = None,
                        clientSecret: Option[String] = None,
                        authURL: Option[String] = None)

object CommandParser {

  val parser: OptionParser[ParserConfig] = new scopt.OptionParser[ParserConfig]("umi") {
    head("ANG Service UMI", "1.0")
    cmd("populate")
      .text("Populates dialect instances ")
      .children {
        arg[String]("angURL")
          .action((u, c) => c.copy(angURL = Option(u)))
          .text("The address to the endpoint on ANG-service")
        arg[String]("output")
          .action((path, c) => c.copy(output = Option(Fs.syncFile(path))))
          .text("The path to the folder where the report will be dumped")
        opt[String]("authURL")
          .action((url, c) => c.copy(authURL = Option(url)))
          .text("An url to an endpoint of authentication service. AuthURL must be used altogether with clientID and clientSecret. Default is no auth")
        opt[String]("clientID")
          .action((id, c) => c.copy(clientID = Option(id)))
          .text("An identification of a user register in Authentication Service. ClientID must be used altogether with authURL and clientSecret. Default is no auth")
        opt[String]("clientSecret")
          .action((s, c) => c.copy(clientSecret = Option(s)))
          .text("The password of a clientID. ClientSecret must be used altogether with authURL and clientID. Default is no auth")
        opt[Int]('i', "instances")
          .action((i, c) => c.copy(instances = i))
          .text("The number of dialect instances you want to generate on each container. Default is 1000")
        opt[Int]('c', "cardinality")
          .action((i, c) => c.copy(cardinality = i))
          .text("The maximum length of generated lists and maps. Default is 3")
        opt[Int]('p', "parallelism")
          .action((p, c) => c.copy(parallelism = p))
          .text("The number of concurrent clients. Default is 10")
        opt[Long]('s', "seed")
          .action((l, c) => c.copy(seed = l))
          .text("The seed used when generating files. Default is random")
      }

    checkConfig(c => {
      val builder: StringBuilder = StringBuilder.newBuilder
      if (c.angURL.isEmpty) builder.append("Missing <angURL> URL\n")
      if (c.output.isEmpty) builder.append("Missing <output> path\n")
      if (c.parallelism < 1 || c.parallelism > 32) builder.append("Parallelism can be minimum 1 and maximum 32 \n")
      if (!validateAuthParams(c))
        builder.append(
          "One or two arguments of authURL, clientID and clientSecret are missing. " +
            "They must be used altogether in order to use auth\n"
        )
      if (builder.isEmpty) success else failure(builder.toString())
    })
  }

  private def validateAuthParams(c: ParserConfig): Boolean = {
    // if any is defined check if all of auth params are defined
    if (c.authURL.isDefined || c.clientSecret.isDefined || c.clientID.isDefined) {
      c.authURL.isDefined && c.clientSecret.isDefined && c.clientID.isDefined
    } else {
      true // else all of them aren't defined and is okay!
    }
  }

  def parse(args: Array[String]): Option[ParserConfig] = parser.parse(args, ParserConfig())
}
