package ang.umi.common

import akka.actor.{Actor, Cancellable, Props}
import akka.http.scaladsl.model.headers.{Authorization, OAuth2BearerToken}
import akka.pattern.pipe
import akka.http.scaladsl.model.{HttpHeader, MediaTypes}
import akka.http.scaladsl.unmarshalling.Unmarshal
import akka.http.scaladsl.{Http, HttpExt}
import akka.stream.{ActorMaterializer, ActorMaterializerSettings}
import ang.umi.common.TokenHolder.GetToken
import spray.json._

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.concurrent.duration._
import scala.util.{Failure, Success, Try}

case class Token(access_token: String, token_type: String)

object MyJsonProtocol extends DefaultJsonProtocol {
  implicit val tokenFormat: RootJsonFormat[Token] = jsonFormat2(Token)
}

class TokenHolder(authServiceUrl: String, clientID: String, clientSecret: String) extends Actor with AppConst {
  import MyJsonProtocol._

  final implicit val materializer: ActorMaterializer = ActorMaterializer(ActorMaterializerSettings(context.system))
  final implicit val http: HttpExt                   = Http(context.system)
  val interval: FiniteDuration                       = 60 seconds
  val scheduler: Cancellable                         = context.system.scheduler.schedule(interval, interval, self, GetToken)

  override def receive: Receive = {
    case GetToken => getToken.pipeTo(sender)
  }

  private def getToken: Future[Unit] = {
    for {
      r <- Request.post(authServiceUrl,
                        s"client_id=$clientID&client_secret=$clientSecret&grant_type=client_credentials",
                        MediaTypes.`application/x-www-form-urlencoded`)
      t <- Unmarshal(r.entity).to[String]
    } yield {
      Try(t.parseJson.convertTo[Token]) match {
        case Success(token) => TokenHolder.addToken(token.access_token)
        case Failure(e)     => println(e.getMessage)
      }
    }
  }
}

object TokenHolder {

  object GetToken

  private var token: Option[String] = None

  def addToken(t: String): Unit = token = Option(t)

  def getToken: Option[String] = token

  def getTokenHeader: Option[HttpHeader] = getToken.map(t => Authorization(OAuth2BearerToken(t)))

  def props(authServiceUrl: String, clientID: String, clientSecret: String) =
    Props(new TokenHolder(authServiceUrl, clientID, clientSecret))
}
