package ang.umi.orchestrator

import akka.actor.ActorRef
import akka.http.scaladsl.model.HttpResponse
import akka.http.scaladsl.unmarshalling.Unmarshal
import akka.stream.scaladsl.{Interleave, Source}
import akka.stream.{Materializer, ThrottleMode}
import akka.util.Timeout
import amf.plugins.document.vocabularies.model.document.Dialect
import aml.gen.GenDoc
import ang.umi.common.{AppMessage, PopulationDone}
import ang.umi.report.{Event, Report, RequestTracker}
import ang.umi.worker.Worker.{PostANG, PostANGResult}
import org.scalacheck.Gen
import org.scalacheck.Gen.Parameters
import org.scalacheck.rng.Seed
import org.yaml.render.YamlRender

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.concurrent.duration.{FiniteDuration, _}

case class Orchestrator(router: ActorRef,
                        instances: Int,
                        seed: Long,
                        cardinality: Int,
                        isLimited: Boolean,
                        parallelism: Int,
                        report: Report)(implicit mat: Materializer) {

  val rate                     = 100
  val interval: FiniteDuration = 60 seconds

  def start(dialects: List[(String, Dialect)]): Future[AppMessage] = {
    implicit val timeout: Timeout = 3 minutes

    val sources = dialects.map(
      t =>
        Source(
          Gen
            .infiniteStream(GenDoc.doc(t._2))
            .pureApply(Parameters.default.withSize(cardinality), Seed(seed))
            .map((t._1, _))
        ).take(instances)
    )

    val source = sources match {
      case Nil                 => Source.empty
      case head :: Nil         => head
      case head :: last :: Nil => head.interleave(last, 1)
      case x :: y :: tail      => Source.combine(x, y, tail: _*)(_ => Interleave(sources.size, 1))
      case _                   => Source.empty
    }

    val limitedSource = if (isLimited) source.throttle(rate, interval, parallelism, ThrottleMode.shaping) else source

    limitedSource.zipWithIndex
      .map({
        case (t, index) =>
          val yaml = YamlRender.render(t._2)
          report.addEvent(Event(index, "URL", t._1))
          report.addEvent(Event(index, "Yaml sent", yaml))
          PostANG(t._1, yaml, index)
      })
      .ask[PostANGResult](parallelism)(router)
      .runForeach {
        case PostANGResult(index, response) => handleANGResult(index, response)
      }
      .map(_ => PopulationDone())
  }

  private def handleANGResult(index: Long, response: HttpResponse): Unit = {
    RequestTracker
      .getTime(index)
      .foreach(time => report.addEvent(Event(index, "Request Time (millis)", time.toString)))
    println(s"#$index - Response: ${response.status}")
    Unmarshal(response.entity)
      .to[String]
      .foreach(body => {
        report.addEvent(Event(index, "Response Body", body))
        report.addEvent(Event(index, "Response Status", response.status.value))
        if (response.status.isFailure()) println(body)
      })
  }
}
