package ang.umi.extractor

import akka.actor.{Actor, Props}
import akka.http.scaladsl.model.HttpHeader
import akka.http.scaladsl.unmarshalling.Unmarshal
import akka.http.scaladsl.{Http, HttpExt}
import akka.pattern._
import akka.stream.{ActorMaterializer, ActorMaterializerSettings}
import amf.core.model.document.BaseUnit
import amf.plugins.document.vocabularies.model.document.{Dialect, DialectInstance}
import amf.plugins.document.vocabularies.model.domain.DialectDomainElement
import ang.umi.common.KnownMediaTypes.`application/yaml`
import ang.umi.common._
import ang.umi.extractor.Extractor.StartExtractor
import ang.umi.orchestrator.Orchestrator

import scala.annotation.tailrec
import scala.collection.mutable.ListBuffer
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

class Extractor(orchestrator: Orchestrator) extends Actor with AmfOps with AppConst {

  final private implicit val materializer: ActorMaterializer = ActorMaterializer(
    ActorMaterializerSettings(context.system))
  final private implicit val http: HttpExt = Http(context.system)

  override def receive: Receive = {
    case StartExtractor(url) => startExtraction(url) pipeTo sender
  }

  private def startExtraction(url: String): Future[AppMessage] = {
    implicit val authHeaders: List[HttpHeader] = TokenHolder.getTokenHeader.map(List(_)).getOrElse(List.empty)
    for {
      systemRequest <- Request.get(s"$url/$SYSTEM?$FIRST_PAGE", `application/yaml`, authHeaders)
      system        <- Unmarshal(systemRequest.entity).to[String]
      _             <- registerSystemContainer(system)
      members       <- getAllMembers(system, systemRequest.headers)
      dialects      <- Future.sequence(members.map(getMembersDialect))
      response      <- startPosting(dialects)
    } yield {
      response
    }
  }

  private def registerSystemContainer(containerConf: String)(implicit authHeader: List[HttpHeader]): Future[BaseUnit] = {
    val dialectUrl = getSchemaURL(containerConf)
    for {
      r              <- Request.get(dialectUrl, `application/yaml`, authHeader)
      dialectContent <- Unmarshal(r.entity).to[String]
      _              <- analyzeVocabulary(dialectContent)
      dialect        <- parse(dialectUrl, dialectContent)
    } yield {
      dialect
    }
  }

  private def analyzeVocabulary(strDialect: String)(implicit authHeader: List[HttpHeader]): Future[Unit] = {
    strDialect.lines.toList
      .filter(_.contains(USES_KEY(1)))
      .map(_.replace(s"${USES_KEY(1)}:", "").trim)
      .headOption
      .map { url =>
        for {
          r1                <- Request.get(url, `application/yaml`, authHeader)
          vocabularyContent <- Unmarshal(r1.entity).to[String]
          _                 <- parse(url, vocabularyContent)
        } yield {}
      }
      .getOrElse(Future.unit)
  }

  private def getSchemaURL(container: String): String =
    container.lines.toList.filter(_.startsWith(SCHEMA)).map(_.replace(s"$SCHEMA:", "").trim).head

  private def getAllMembers(container: String, headers: Seq[HttpHeader])(
      implicit authHeader: List[HttpHeader]): Future[Seq[String]] = {
    val next = getHeader(headers, LINK, NEXT_LINK)
    for {
      di          <- parse(UMI_URL, container).mapTo[DialectInstance]
      allDialects <- getAllPages(next)
    } yield {
      val containers = getMembersFromDialect(di) ++ allDialects.flatMap(getMembersFromDialect)
      containers.filterNot(m => m.contains(VOCABULARY) || m.contains(SCHEMA) || m.contains(SYSTEM))
    }
  }

  private def getAllPages(next: Option[HttpHeader])(
      implicit authHeader: List[HttpHeader]): Future[Seq[DialectInstance]] = {
    next match {
      case Some(header) =>
        val uri = header.value().split(";").head.drop(1).dropRight(1)
        val eventualUnit = for {
          r       <- Request.get(uri, `application/yaml`, authHeader)
          content <- Unmarshal(r.entity).to[String]
          di      <- parse(UMI_URL, content).mapTo[DialectInstance]
        } yield {
          (di, getHeader(r.headers, LINK, NEXT_LINK))
        }
        eventualUnit.flatMap(t => getAllPages(t._2))
      case None => Future.successful(Seq.empty)
    }
  }

  private def getHeader(headers: Seq[HttpHeader], header: String, id: String): Option[HttpHeader] =
    headers.filter(_.is(header)).find(_.value().contains(id))

  private def getMembersFromDialect(di: DialectInstance): Seq[String] = {
    di.encodes
      .asInstanceOf[DialectDomainElement]
      .linkProperties
      .find(_._1.contains(MEMBERS))
      .map(_._2.asInstanceOf[Seq[String]])
      .getOrElse(Seq.empty)
  }

  private def getMembersDialect(uri: String)(implicit authHeader: List[HttpHeader]): Future[(String, Dialect)] = {
    for {
      r       <- Request.get(s"$uri?$FIRST_PAGE", `application/yaml`, authHeader)
      content <- Unmarshal(r.entity).to[String]
      dialect <- getDialectFromMember(content).mapTo[Dialect]
    } yield {
      (uri, dialect)
    }
  }

  private def getDialectFromMember(containerConf: String)(implicit authHeader: List[HttpHeader]): Future[BaseUnit] = {
    val dialectUrl = getSchemaURL(containerConf)
    for {
      r              <- Request.get(dialectUrl, `application/yaml`, authHeader)
      dialectContent <- Unmarshal(r.entity).to[String]
      dialect        <- parse(UMI_URL, transformToExternal(dialectContent))
    } yield {
      dialect
    }
  }

  private def transformToExternal(dialect: String): String = {
    if (!dialect.contains(s"$USES:")) return dialect
    if (dialect.contains(s"$EXTERNAL:")) {
      // searching for all uses and adding them to a list
      val lines    = dialect.lines.toList
      var usesList = new ListBuffer[String]()
      var index    = getIndexOfLine(s"${USES_KEY(1)}:", lines)
      var i        = 1
      while (index != -1) {
        usesList += lines(index)
        i += 1
        index = getIndexOfLine(s"${USES_KEY(i)}:", lines)
      }

      // removing uses
      val usesTag     = getIndexOfLine(s"$USES:", lines)
      val withoutUses = lines.slice(0, usesTag) ++ lines.slice(usesTag + i, lines.length)

      // search for external and appends all de uses to it
      val externalIndex = getIndexOfLine(s"$EXTERNAL:", withoutUses)
      val fullDialect = withoutUses.slice(0, externalIndex + 1) ++ usesList ++ withoutUses.slice(externalIndex + 1,
                                                                                                 lines.size)
      fullDialect.mkString("\n")
    } else {
      dialect.replaceFirst(s"$USES:", s"$EXTERNAL:")
    }
  }

  @tailrec private def getIndexOfLine(line: String, list: List[String], i: Int = 0): Int = {
    list match {
      case Nil       => -1
      case x :: tail => if (x.contains(line)) i else getIndexOfLine(line, tail, i + 1)
    }
  }

  private def startPosting(dialects: Seq[(String, Dialect)]): Future[AppMessage] = {
    if (dialects.isEmpty) Future.successful(NoPopulation(NO_POPULATION))
    else orchestrator.start(dialects.toList)
  }
}

object Extractor {
  def props(orchestrator: Orchestrator) = Props(new Extractor(orchestrator))

  case class StartExtractor(uri: String)
}
