package ang.umi.report

import java.text.{DecimalFormat, SimpleDateFormat}
import java.util.Date

import org.apache.commons.math3.stat.descriptive.SynchronizedSummaryStatistics

/**
  * Generate Summary Statistics and Frequency Distributions
  */
case class PerformanceStat(name: String, values: List[Double], success: Int, failed: Int) {

  private val newLine: String = System.getProperty("line.separator")

  private def generateSummaryStatistics(): SynchronizedSummaryStatistics = {
    val summaryStatistics = new SynchronizedSummaryStatistics()
    values.foreach(summaryStatistics.addValue)
    summaryStatistics
  }

  def getAggregateReport: String = {

    def getDateTime: String = {
      val date        = new Date()
      val timePattern = "yyyy-MM-dd HH:mm:ss"
      val formatter   = new SimpleDateFormat(timePattern)
      formatter.format(date)
    }

    val recordCounter = generateSummaryStatistics().getN
    val sb            = new StringBuffer()

    sb.append(newLine).append(name).append(newLine)
    sb.append("----------------------------------------------------").append(newLine).append(newLine)
    sb.append("Date: ").append(getDateTime).append(newLine)
    sb.append("# of calls: ").append(recordCounter).append(newLine).append(newLine)
    sb.append(getSummaryStatisticsReport).append(newLine)
    sb.append(getPercentile).append(newLine).append(newLine)

    sb.toString
  }

  private def getSummaryStatisticsReport: String = {
    val summaryStatistics = generateSummaryStatistics()
    val decimalFormat     = new DecimalFormat("####,###.00")
    val aMax              = decimalFormat.format(summaryStatistics.getMax) + "ms"
    val aMin              = decimalFormat.format(summaryStatistics.getMin) + "ms"
    val aMean             = decimalFormat.format(summaryStatistics.getMean) + "ms"
    val aStd              = decimalFormat.format(summaryStatistics.getStandardDeviation) + "ms"

    val sb = new StringBuffer()

    sb.append("Summary Statistics").append(newLine)
    sb.append("Succeed: ").append(success).append(newLine)
    sb.append("Failed: ").append(failed).append(newLine)
    sb.append("Max: ").append(aMax).append(newLine)
    sb.append("Min: ").append(aMin).append(newLine)
    sb.append("Mean: ").append(aMean).append(newLine)
    sb.append("Standard Deviation: ").append(aStd).append(newLine)

    sb.toString
  }

  private def roundUp(d: Double) = math.ceil(d).toInt

  private def getPercentile: String = {
    if (values.nonEmpty) {
      val percentage   = List(25.0, 50.0, 75.0, 100.0)
      val sortedValues = values.sorted
      val pRange       = percentage.map(p => p -> roundUp((p / 100) * sortedValues.length))

      val builder = pRange.foldLeft(StringBuilder.newBuilder)((acc, p) =>
        acc.append(s"${p._1}%: <${sortedValues(p._2 - 1)}ms (${p._2}/${sortedValues.length}) \n"))

      s"""
         |Percentile
         |${builder.mkString}
      """.stripMargin
    } else
      "Percentile: <empty>"
  }
}
