package ang.umi.commands

import akka.actor.{ActorRef, ActorSystem}
import akka.pattern.ask
import akka.stream.ActorMaterializer
import akka.util.Timeout
import ang.umi.common.TokenHolder.GetToken
import ang.umi.common.{AppMessage, TokenHolder}
import ang.umi.extractor.Extractor
import ang.umi.extractor.Extractor.StartExtractor
import ang.umi.orchestrator.Orchestrator
import ang.umi.report.Report

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.concurrent.duration._

case class PopulateCommand private (cfg: ParserConfig, report: Report) {

  implicit final private val system: ActorSystem             = ActorSystem("ang-service-benchmark")
  implicit final private val materializer: ActorMaterializer = ActorMaterializer()

  def execute(): Future[AppMessage] = {
    implicit val timeout: Timeout = 2 minutes

    val authActor: Option[ActorRef] = for {
      url          <- cfg.authURL
      clientId     <- cfg.clientID
      clientSecret <- cfg.clientSecret
    } yield {
      system.actorOf(TokenHolder.props(url, clientId, clientSecret), "token")
    }

    val angURL = cfg.angURL.get
    val orchestrator =
      Orchestrator(angURL, cfg.instances, cfg.seed, cfg.cardinality, authActor.isDefined, cfg.parallelism, report)
    val extractor = system.actorOf(Extractor.props(orchestrator), "extractor")
    authActor.map(_ ? GetToken).getOrElse(Future.unit).flatMap(_ => startExtraction(angURL, extractor))
  }

  private def startExtraction(url: String, extractor: ActorRef): Future[AppMessage] = {
    implicit val timeout: Timeout = new Timeout(99 day)
    extractor.ask(StartExtractor(url)).mapTo[AppMessage]
  }
}
