package ang.umi.orchestrator

import java.net.URL

import akka.actor.ActorSystem
import akka.http.scaladsl.Http
import akka.http.scaladsl.model.HttpEntity.Strict
import akka.http.scaladsl.model._
import akka.stream.scaladsl.{Flow, Interleave, Sink, Source}
import akka.stream.{Materializer, ThrottleMode}
import akka.util.Timeout
import amf.plugins.document.vocabularies.model.document.Dialect
import aml.gen.GenDoc
import ang.umi.common.KnownContentTypes.`application/yaml`
import ang.umi.common._
import ang.umi.report.Report
import org.scalacheck.Gen
import org.scalacheck.Gen.Parameters
import org.scalacheck.rng.Seed
import org.yaml.render.YamlRender

import scala.collection.immutable
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.concurrent.duration.{FiniteDuration, _}
import scala.util.{Failure, Success, Try}

case class Orchestrator(angURL: String,
                        instances: Int,
                        seed: Long,
                        cardinality: Int,
                        isLimited: Boolean,
                        parallelism: Int,
                        report: Report)(implicit mat: Materializer, system: ActorSystem) {

  val rate = 100
  // In order to execute every pool al least once
  val warmUps: Int             = HttpConnection.biggestPowerOf2(parallelism) * 2
  val interval: FiniteDuration = 60 seconds

  def start(dialects: List[(String, Dialect)]): Future[AppMessage] = {
    implicit val timeout: Timeout = 3 minutes

    warming(dialects) {
      benchmark(dialects)
    }
  }

  private def warming[T](dialects: List[(String, Dialect)])(next: => Future[T]): Future[T] = {
    println("Warming up...")
    source(dialects)
      .take(warmUps)
      .zipWithIndex
      .map {
        case (t, index) =>
          val yaml    = YamlRender.render(t._2)
          val headers = TokenHolder.getTokenHeader.map(List(_)).getOrElse(List.empty)
          val entity  = HttpEntity(`application/yaml`, yaml)
          HttpRequest(HttpMethods.POST, uri = t._1, entity = entity, headers = headers) -> Lap(index, t._1)
      }
      .via(getConnection)
      .statefulMapConcat(warmUpSummarization)
      .runWith(Sink.ignore)
      .flatMap(_ => next)
  }

  private def benchmark(dialects: List[(String, Dialect)]): Future[AppMessage] = {
    println("Benchmarking...")
    source(dialects).zipWithIndex
      .map {
        case (t, index) =>
          val yaml    = YamlRender.render(t._2)
          val headers = TokenHolder.getTokenHeader.map(List(_)).getOrElse(List.empty)
          val entity  = HttpEntity(`application/yaml`, yaml)
          HttpRequest(HttpMethods.POST, uri = t._1, entity = entity, headers = headers) -> Lap(index, t._1)
      }
      .via(getConnection)
      .statefulMapConcat(summarization)
      .runWith(Sink.last)
      .map(_ => PopulationDone())
      .recover {
        case e => PopulationError(e)
      }
  }

  private def source(dialects: List[(String, Dialect)]) = {
    val sources = dialects.map(
      t =>
        Source(
          Gen
            .infiniteStream(GenDoc.doc(t._2))
            .pureApply(Parameters.default.withSize(cardinality), Seed(seed))
            .map((t._1, _))
        ).take(instances)
    )

    val source = sources match {
      case head :: Nil         => head
      case head :: last :: Nil => head.interleave(last, 1)
      case x :: y :: tail      => Source.combine(x, y, tail: _*)(_ => Interleave(sources.size, 1))
      case Nil | _             => Source.empty
    }

    if (isLimited) source.throttle(rate, interval, parallelism, ThrottleMode.shaping) else source
  }

  private def warmUpSummarization: () => ((Try[HttpResponse], Lap)) => immutable.Iterable[Unit] = { () => t =>
    {
      t._2.stop
      t._1 match {
        case Success(r) =>
          r.discardEntityBytes()
          println(s"Warmup #${t._2.index} -> url: ${t._2.url}, time: ${t._2.elapsed}")
        case Failure(exception) =>
          println(s"Warmup #${t._2.index} Exception: ${exception.toString}")
      }
      immutable.Iterable(Unit)
    }
  }

  private def summarization: () => ((Try[HttpResponse], Lap)) => immutable.Iterable[Report] = { () => t =>
    {
      t._2.stop
      t._1 match {
        case Success(r) if r.status.isSuccess() =>
          r.discardEntityBytes()
          println(s"#${t._2.index} Success: ${t._2.url}, time: ${t._2.elapsed}")
        case Success(r) =>
          r.discardEntityBytes()
          println(
            s"#${t._2.index} Failed: ${t._2.url} -> ${r.entity.asInstanceOf[Strict].getData().utf8String}, time: ${t._2.elapsed}")
        case Failure(exception) =>
          println(s"#${t._2.index} Exception: ${exception.toString}")
      }
      immutable.Iterable(report.lap(t))
    }
  }

  private def getConnection[T]: Flow[(HttpRequest, T), (Try[HttpResponse], T), Http.HostConnectionPool] = {
    val url = new URL(angURL)
    if (url.getPort == -1 && url.getProtocol.toLowerCase == "https")
      HttpConnection.secureConnection(url.getHost, parallelism)
    else if (url.getPort == -1 && url.getProtocol.toLowerCase == "http")
      HttpConnection.connection(url.getHost, concurrency = parallelism)
    else HttpConnection.connection(url.getHost, url.getPort, parallelism)
  }
}
