/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.apache.xerces.impl.xs.opti;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.xerces.impl.XMLDocumentFragmentScannerImpl;
import org.apache.xerces.impl.XMLErrorReporter;
import org.apache.xerces.impl.dv.InvalidDatatypeValueException;
import org.apache.xerces.impl.dv.ValidationContext;
import org.apache.xerces.impl.dv.xs.DecimalDV;
import org.apache.xerces.impl.dv.xs.TypeValidator;
import org.apache.xerces.impl.xs.SchemaSymbols;
import org.apache.xerces.impl.xs.util.XS11TypeHelper;
import org.apache.xerces.util.XMLAttributesImpl;
import org.apache.xerces.util.XMLChar;
import org.apache.xerces.xni.Augmentations;
import org.apache.xerces.xni.NamespaceContext;
import org.apache.xerces.xni.QName;
import org.apache.xerces.xni.XMLAttributes;
import org.apache.xerces.xni.XMLLocator;
import org.apache.xerces.xni.XMLString;
import org.apache.xerces.xni.XNIException;
import org.apache.xerces.xni.parser.XMLEntityResolver;
import org.apache.xerces.xni.parser.XMLInputSource;
import org.apache.xerces.xni.parser.XMLParserConfiguration;
import org.apache.xerces.xs.datatypes.XSDecimal;
import org.w3c.dom.Document;

/**
 * Shadowed class so that {@link SchemaDOMParser} without the fSupportedVersion set works.
 * This is used by the SDK but there should be a default for fSupportedVersion in {@link SchemaDOMParser#isSchemaLangVersionAllowsExclude}.
 */
public class SchemaDOMParser extends DefaultXMLDocumentHandler {
    public static final String ERROR_REPORTER = "http://apache.org/xml/properties/internal/error-reporter";
    public static final String GENERATE_SYNTHETIC_ANNOTATION = "http://apache.org/xml/features/generate-synthetic-annotations";
    protected XMLLocator fLocator;
    protected NamespaceContext fNamespaceContext = null;

    private static final TypeValidator DECIMAL_DV = new DecimalDV();
    SchemaDOM schemaDOM;
    XMLParserConfiguration config;
    private ElementImpl fCurrentAnnotationElement;
    private int fAnnotationDepth = -1;
    private int fInnerAnnotationDepth = -1;
    private int fDepth = -1;
    XMLErrorReporter fErrorReporter;
    private boolean fGenerateSyntheticAnnotation = false;
    private BooleanStack fHasNonSchemaAttributes = new BooleanStack();
    private BooleanStack fSawAnnotation = new BooleanStack();
    private XMLAttributes fEmptyAttr = new XMLAttributesImpl();
    private XSDecimal fSupportedVersion;
    private int fIgnoreDepth = -1;
    private boolean fPerformConditionalInclusion = true;
    private SchemaConditionalIncludeHelper schemaCondlInclHelper = new SchemaConditionalIncludeHelper();

    private static final XSDecimal SUPPORTED_VERSION_1_0 = getSupportedVersion("1.0");

    public SchemaDOMParser(XMLParserConfiguration config) {
        this.config = config;
        config.setDocumentHandler(this);
        config.setDTDHandler(this);
        config.setDTDContentModelHandler(this);
        this.fSupportedVersion = SUPPORTED_VERSION_1_0;
    }

    public void startDocument(XMLLocator locator, String encoding, NamespaceContext namespaceContext, Augmentations augs) throws XNIException {
        this.fErrorReporter = (XMLErrorReporter)this.config.getProperty("http://apache.org/xml/properties/internal/error-reporter");
        this.fGenerateSyntheticAnnotation = this.config.getFeature("http://apache.org/xml/features/generate-synthetic-annotations");
        this.fHasNonSchemaAttributes.clear();
        this.fSawAnnotation.clear();
        this.schemaDOM = new SchemaDOM();
        this.fCurrentAnnotationElement = null;
        this.fAnnotationDepth = -1;
        this.fInnerAnnotationDepth = -1;
        this.fDepth = -1;
        this.fIgnoreDepth = -1;
        this.fLocator = locator;
        this.fNamespaceContext = namespaceContext;
        this.schemaDOM.setDocumentURI(locator.getExpandedSystemId());
    }

    public void endDocument(Augmentations augs) throws XNIException {
    }

    public void comment(XMLString text, Augmentations augs) throws XNIException {
        if (this.fAnnotationDepth > -1 && this.fIgnoreDepth == -1) {
            this.schemaDOM.comment(text);
        }

    }

    public void processingInstruction(String target, XMLString data, Augmentations augs) throws XNIException {
        if (this.fAnnotationDepth > -1 && this.fIgnoreDepth == -1) {
            this.schemaDOM.processingInstruction(target, data);
        }

    }

    public void characters(XMLString text, Augmentations augs) throws XNIException {
        if (this.fIgnoreDepth <= -1) {
            if (this.fInnerAnnotationDepth == -1) {
                for(int i = text.offset; i < text.offset + text.length; ++i) {
                    if (!XMLChar.isSpace(text.ch[i])) {
                        String txt = new String(text.ch, i, text.length + text.offset - i);
                        this.fErrorReporter.reportError(this.fLocator, "http://www.w3.org/TR/xml-schema-1", "s4s-elt-character", new Object[]{txt}, (short)1);
                        break;
                    }
                }
            } else {
                this.schemaDOM.characters(text);
            }

        }
    }

    public void startElement(QName element, XMLAttributes attributes, Augmentations augs) throws XNIException {
        ++this.fDepth;
        if (this.fPerformConditionalInclusion) {
            if (this.fIgnoreDepth > -1) {
                ++this.fIgnoreDepth;
                return;
            }

            if (this.fDepth > -1) {
                this.checkVersionControlAttributes(element, attributes);
                if (this.fIgnoreDepth > -1) {
                    return;
                }
            }
        }

        if (this.fAnnotationDepth == -1) {
            if (element.uri == SchemaSymbols.URI_SCHEMAFORSCHEMA && element.localpart == SchemaSymbols.ELT_ANNOTATION) {
                if (this.fGenerateSyntheticAnnotation) {
                    if (this.fSawAnnotation.size() > 0) {
                        this.fSawAnnotation.pop();
                    }

                    this.fSawAnnotation.push(true);
                }

                this.fAnnotationDepth = this.fDepth;
                this.schemaDOM.startAnnotation(element, attributes, this.fNamespaceContext);
                this.fCurrentAnnotationElement = this.schemaDOM.startElement(element, attributes, this.fLocator.getLineNumber(), this.fLocator.getColumnNumber(), this.fLocator.getCharacterOffset());
                return;
            }

            if (element.uri == SchemaSymbols.URI_SCHEMAFORSCHEMA && this.fGenerateSyntheticAnnotation) {
                this.fSawAnnotation.push(false);
                this.fHasNonSchemaAttributes.push(this.hasNonSchemaAttributes(element, attributes));
            }
        } else {
            if (this.fDepth != this.fAnnotationDepth + 1) {
                this.schemaDOM.startAnnotationElement(element, attributes);
                return;
            }

            this.fInnerAnnotationDepth = this.fDepth;
            this.schemaDOM.startAnnotationElement(element, attributes);
        }

        this.schemaDOM.startElement(element, attributes, this.fLocator.getLineNumber(), this.fLocator.getColumnNumber(), this.fLocator.getCharacterOffset());
    }

    public void emptyElement(QName element, XMLAttributes attributes, Augmentations augs) throws XNIException {
        if (this.fPerformConditionalInclusion) {
            if (this.fIgnoreDepth > -1) {
                return;
            }

            if (this.fDepth > -1) {
                boolean vcExclude = this.checkVersionControlAttributes(element, attributes);
                if (this.fIgnoreDepth > -1) {
                    if (vcExclude) {
                        --this.fIgnoreDepth;
                    }

                    return;
                }
            }
        }

        if (this.fGenerateSyntheticAnnotation && this.fAnnotationDepth == -1 && element.uri == SchemaSymbols.URI_SCHEMAFORSCHEMA && element.localpart != SchemaSymbols.ELT_ANNOTATION && this.hasNonSchemaAttributes(element, attributes)) {
            this.schemaDOM.startElement(element, attributes, this.fLocator.getLineNumber(), this.fLocator.getColumnNumber(), this.fLocator.getCharacterOffset());
            attributes.removeAllAttributes();
            String schemaPrefix = this.fNamespaceContext.getPrefix(SchemaSymbols.URI_SCHEMAFORSCHEMA);
            String annRawName = schemaPrefix.length() == 0 ? SchemaSymbols.ELT_ANNOTATION : schemaPrefix + ':' + SchemaSymbols.ELT_ANNOTATION;
            this.schemaDOM.startAnnotation(annRawName, attributes, this.fNamespaceContext);
            String elemRawName = schemaPrefix.length() == 0 ? SchemaSymbols.ELT_DOCUMENTATION : schemaPrefix + ':' + SchemaSymbols.ELT_DOCUMENTATION;
            this.schemaDOM.startAnnotationElement(elemRawName, attributes);
            this.schemaDOM.charactersRaw("SYNTHETIC_ANNOTATION");
            this.schemaDOM.endSyntheticAnnotationElement(elemRawName, false);
            this.schemaDOM.endSyntheticAnnotationElement(annRawName, true);
            this.schemaDOM.endElement();
        } else {
            if (this.fAnnotationDepth == -1) {
                if (element.uri == SchemaSymbols.URI_SCHEMAFORSCHEMA && element.localpart == SchemaSymbols.ELT_ANNOTATION) {
                    this.schemaDOM.startAnnotation(element, attributes, this.fNamespaceContext);
                }
            } else {
                this.schemaDOM.startAnnotationElement(element, attributes);
            }

            ElementImpl newElem = this.schemaDOM.emptyElement(element, attributes, this.fLocator.getLineNumber(), this.fLocator.getColumnNumber(), this.fLocator.getCharacterOffset());
            if (this.fAnnotationDepth == -1) {
                if (element.uri == SchemaSymbols.URI_SCHEMAFORSCHEMA && element.localpart == SchemaSymbols.ELT_ANNOTATION) {
                    this.schemaDOM.endAnnotation(element, newElem);
                }
            } else {
                this.schemaDOM.endAnnotationElement(element);
            }

        }
    }

    public void endElement(QName element, Augmentations augs) throws XNIException {
        if (this.fIgnoreDepth == -1) {
            if (this.fAnnotationDepth > -1) {
                if (this.fInnerAnnotationDepth == this.fDepth) {
                    this.fInnerAnnotationDepth = -1;
                    this.schemaDOM.endAnnotationElement(element);
                    this.schemaDOM.endElement();
                } else if (this.fAnnotationDepth == this.fDepth) {
                    this.fAnnotationDepth = -1;
                    this.schemaDOM.endAnnotation(element, this.fCurrentAnnotationElement);
                    this.schemaDOM.endElement();
                } else {
                    this.schemaDOM.endAnnotationElement(element);
                }
            } else {
                if (element.uri == SchemaSymbols.URI_SCHEMAFORSCHEMA && this.fGenerateSyntheticAnnotation) {
                    boolean value = this.fHasNonSchemaAttributes.pop();
                    boolean sawann = this.fSawAnnotation.pop();
                    if (value && !sawann) {
                        String schemaPrefix = this.fNamespaceContext.getPrefix(SchemaSymbols.URI_SCHEMAFORSCHEMA);
                        String annRawName = schemaPrefix.length() == 0 ? SchemaSymbols.ELT_ANNOTATION : schemaPrefix + ':' + SchemaSymbols.ELT_ANNOTATION;
                        this.schemaDOM.startAnnotation(annRawName, this.fEmptyAttr, this.fNamespaceContext);
                        String elemRawName = schemaPrefix.length() == 0 ? SchemaSymbols.ELT_DOCUMENTATION : schemaPrefix + ':' + SchemaSymbols.ELT_DOCUMENTATION;
                        this.schemaDOM.startAnnotationElement(elemRawName, this.fEmptyAttr);
                        this.schemaDOM.charactersRaw("SYNTHETIC_ANNOTATION");
                        this.schemaDOM.endSyntheticAnnotationElement(elemRawName, false);
                        this.schemaDOM.endSyntheticAnnotationElement(annRawName, true);
                    }
                }

                this.schemaDOM.endElement();
            }
        } else {
            --this.fIgnoreDepth;
        }

        --this.fDepth;
    }

    private boolean hasNonSchemaAttributes(QName element, XMLAttributes attributes) {
        int length = attributes.getLength();

        for(int i = 0; i < length; ++i) {
            String uri = attributes.getURI(i);
            if (uri != null && uri != SchemaSymbols.URI_SCHEMAFORSCHEMA && uri != NamespaceContext.XMLNS_URI && (uri != NamespaceContext.XML_URI || attributes.getQName(i) != SchemaSymbols.ATT_XML_LANG || element.localpart != SchemaSymbols.ELT_SCHEMA)) {
                return true;
            }
        }

        return false;
    }

    public void ignorableWhitespace(XMLString text, Augmentations augs) throws XNIException {
        if (this.fAnnotationDepth != -1 && this.fIgnoreDepth == -1) {
            this.schemaDOM.characters(text);
        }

    }

    public void startCDATA(Augmentations augs) throws XNIException {
        if (this.fAnnotationDepth != -1 && this.fIgnoreDepth == -1) {
            this.schemaDOM.startAnnotationCDATA();
        }

    }

    public void endCDATA(Augmentations augs) throws XNIException {
        if (this.fAnnotationDepth != -1 && this.fIgnoreDepth == -1) {
            this.schemaDOM.endAnnotationCDATA();
        }

    }

    public Document getDocument() {
        return this.schemaDOM;
    }

    public void setFeature(String featureId, boolean state) {
        this.config.setFeature(featureId, state);
    }

    public boolean getFeature(String featureId) {
        return this.config.getFeature(featureId);
    }

    public void setProperty(String propertyId, Object value) {
        this.config.setProperty(propertyId, value);
    }

    public Object getProperty(String propertyId) {
        return this.config.getProperty(propertyId);
    }

    public void setEntityResolver(XMLEntityResolver er) {
        this.config.setEntityResolver(er);
    }

    public void parse(XMLInputSource inputSource) throws IOException {
        this.config.parse(inputSource);
    }

    public void reset() {
        ((SchemaParsingConfig)this.config).reset();
    }

    public void resetNodePool() {
        ((SchemaParsingConfig)this.config).resetNodePool();
    }

    public void setSupportedVersion(XSDecimal version) {
        this.fSupportedVersion = version;
    }

    private boolean checkVersionControlAttributes(QName element, XMLAttributes attributes) {
        boolean isIgnoreSchemaComponent = false;
        BigDecimal minVer = null;
        BigDecimal maxVer = null;
        List typeAvailableList = null;
        List typeUnavailableList = null;
        List facetAvailableList = null;
        List facetUnavailableList = null;

        for(int attrIdx = 0; attrIdx < attributes.getLength(); ++attrIdx) {
            if (SchemaSymbols.URI_SCHEMAVERSION.equals(attributes.getURI(attrIdx))) {
                String attrLocalName = attributes.getLocalName(attrIdx);
                String attrValue = attributes.getValue(attrIdx);
                if (SchemaSymbols.ATT_MINVERSION.equals(attrLocalName)) {
                    try {
                        minVer = new BigDecimal(attrValue);
                    } catch (NumberFormatException var16) {
                        this.fErrorReporter.reportError("http://www.w3.org/TR/xml-schema-1", "s4s-att-invalid-value", new Object[]{element.localpart, attrLocalName, var16.getMessage()}, (short)1);
                    }
                } else if (SchemaSymbols.ATT_MAXVERSION.equals(attrLocalName)) {
                    try {
                        maxVer = new BigDecimal(attrValue);
                    } catch (NumberFormatException var15) {
                        this.fErrorReporter.reportError("http://www.w3.org/TR/xml-schema-1", "s4s-att-invalid-value", new Object[]{element.localpart, attrLocalName, var15.getMessage()}, (short)1);
                    }
                } else if (SchemaSymbols.ATT_TYPEAVAILABLE.equals(attrLocalName)) {
                    typeAvailableList = this.tokenizeQNameListString(attrValue);
                } else if (SchemaSymbols.ATT_TYPEUNAVAILABLE.equals(attrLocalName)) {
                    typeUnavailableList = this.tokenizeQNameListString(attrValue);
                } else if (SchemaSymbols.ATT_FACETAVAILABLE.equals(attrLocalName)) {
                    facetAvailableList = this.tokenizeQNameListString(attrValue);
                } else if (SchemaSymbols.ATT_FACETUNAVAILABLE.equals(attrLocalName)) {
                    facetUnavailableList = this.tokenizeQNameListString(attrValue);
                } else {
                    this.fErrorReporter.reportError(this.fLocator, "http://www.w3.org/TR/xml-schema-1", "src-cip.1", new Object[]{attrLocalName}, (short)0);
                }
            }
        }

        boolean minMaxSchemaVerAllowsIgnore = minVer == null && maxVer == null ? false : this.isSchemaLangVersionAllowsExclude(minVer, maxVer);
        boolean typeAvlAllowsIgnore = typeAvailableList == null ? false : this.isTypeAndFacetAvailableAllowsExclude(typeAvailableList, (short)0, (short)2);
        boolean typeUnavlAllowsIgnore = typeUnavailableList == null ? false : this.isTypeAndFacetAvailableAllowsExclude(typeUnavailableList, (short)0, (short)3);
        boolean facetAvlAllowsIgnore = facetAvailableList == null ? false : this.isTypeAndFacetAvailableAllowsExclude(facetAvailableList, (short)1, (short)2);
        boolean facetUnavlAllowsIgnore = facetUnavailableList == null ? false : this.isTypeAndFacetAvailableAllowsExclude(facetUnavailableList, (short)1, (short)3);
        if (!minMaxSchemaVerAllowsIgnore && !typeAvlAllowsIgnore && !typeUnavlAllowsIgnore && !facetAvlAllowsIgnore && !facetUnavlAllowsIgnore) {
            isIgnoreSchemaComponent = false;
        } else {
            isIgnoreSchemaComponent = true;
        }

        return isIgnoreSchemaComponent;
    }

    private boolean isSchemaLangVersionAllowsExclude(BigDecimal minVer, BigDecimal maxVer) {
        boolean minMaxSchemaVerAllowsIgnore = false;
        if (minVer != null && maxVer != null) {
            if (minVer.compareTo(this.fSupportedVersion.getBigDecimal()) > 0 || maxVer.compareTo(this.fSupportedVersion.getBigDecimal()) != 1) {
                ++this.fIgnoreDepth;
                minMaxSchemaVerAllowsIgnore = true;
            }
        } else if (minVer != null && maxVer == null) {
            if (minVer.compareTo(this.fSupportedVersion.getBigDecimal()) > 0) {
                ++this.fIgnoreDepth;
                minMaxSchemaVerAllowsIgnore = true;
            }
        } else if (minVer == null && maxVer != null && maxVer.compareTo(this.fSupportedVersion.getBigDecimal()) != 1) {
            ++this.fIgnoreDepth;
            minMaxSchemaVerAllowsIgnore = true;
        }

        return minMaxSchemaVerAllowsIgnore;
    }

    private boolean isTypeAndFacetAvailableAllowsExclude(List typeOrFacetList, short typeOrFacet, short availaibilityUnavlCheck) {
        boolean typeOrFacetAvlAllowsIgnore = false;
        boolean typeOrFacetUnavlAllowsIgnore = true;
        Iterator iter = typeOrFacetList.iterator();

        while(iter.hasNext()) {
            String typeOrFacetRawValue = (String)iter.next();
            String typeOrFacetLocalName = null;
            String typeOrFacetUri = null;
            if (typeOrFacetRawValue.indexOf(58) != -1) {
                typeOrFacetLocalName = typeOrFacetRawValue.substring(typeOrFacetRawValue.indexOf(58) + 1);
                String typeOrFacetPrefix = typeOrFacetRawValue.substring(0, typeOrFacetRawValue.indexOf(58));
                typeOrFacetUri = this.fNamespaceContext.getURI(typeOrFacetPrefix.intern());
            } else {
                typeOrFacetLocalName = typeOrFacetRawValue;
            }

            if (typeOrFacet == 0) {
                if (availaibilityUnavlCheck == 2 && !this.schemaCondlInclHelper.isTypeSupported(typeOrFacetLocalName, typeOrFacetUri)) {
                    typeOrFacetAvlAllowsIgnore = true;
                    break;
                }

                if (availaibilityUnavlCheck == 3 && !this.schemaCondlInclHelper.isTypeSupported(typeOrFacetLocalName, typeOrFacetUri)) {
                    typeOrFacetUnavlAllowsIgnore = false;
                    break;
                }
            } else if (typeOrFacet == 1) {
                if (availaibilityUnavlCheck == 2 && !this.schemaCondlInclHelper.isFacetSupported(typeOrFacetLocalName, typeOrFacetUri)) {
                    typeOrFacetAvlAllowsIgnore = true;
                    break;
                }

                if (availaibilityUnavlCheck == 3 && !this.schemaCondlInclHelper.isFacetSupported(typeOrFacetLocalName, typeOrFacetUri)) {
                    typeOrFacetUnavlAllowsIgnore = false;
                    break;
                }
            }
        }

        if (availaibilityUnavlCheck == 2) {
            if (typeOrFacetAvlAllowsIgnore) {
                ++this.fIgnoreDepth;
            }

            return typeOrFacetAvlAllowsIgnore;
        } else {
            if (typeOrFacetUnavlAllowsIgnore) {
                ++this.fIgnoreDepth;
            }

            return typeOrFacetUnavlAllowsIgnore;
        }
    }

    private List tokenizeQNameListString(String strValue) {
        StringTokenizer st = new StringTokenizer(strValue, " \n\t\r");
        List stringTokens = new ArrayList(st.countTokens());

        while(st.hasMoreTokens()) {
            String QNameStr = st.nextToken();
            XS11TypeHelper.validateQNameValue(QNameStr, this.fNamespaceContext, this.fErrorReporter);
            stringTokens.add(QNameStr);
        }

        return stringTokens;
    }

    private static final class BooleanStack {
        private int fDepth;
        private boolean[] fData;

        public BooleanStack() {
        }

        public int size() {
            return this.fDepth;
        }

        public void push(boolean value) {
            this.ensureCapacity(this.fDepth + 1);
            this.fData[this.fDepth++] = value;
        }

        public boolean pop() {
            return this.fData[--this.fDepth];
        }

        public void clear() {
            this.fDepth = 0;
        }

        private void ensureCapacity(int size) {
            if (this.fData == null) {
                this.fData = new boolean[32];
            } else if (this.fData.length <= size) {
                boolean[] newdata = new boolean[this.fData.length * 2];
                System.arraycopy(this.fData, 0, newdata, 0, this.fData.length);
                this.fData = newdata;
            }

        }
    }

    private static XSDecimal getSupportedVersion(String version) {
        XSDecimal result = null;

        try {
            // This follows the same strategy as XSDHandler.
            result = (XSDecimal)DECIMAL_DV.getActualValue(version, (ValidationContext)null);
        } catch (InvalidDatatypeValueException var3) {
        }

        return result;
    }
}
