package org.mulesoft.apb.project.client.scala

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.core.client.common.remote.Content
import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.unsafe.PlatformSecrets
import amf.custom.validation.internal.report.generated.ValidationProfileDialectLoader
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.environment.{DependencyFetcher, UnreachableGavException}
import org.mulesoft.apb.project.client.scala.model._
import org.mulesoft.apb.project.internal
import org.mulesoft.apb.project.internal.dependency.UnreachableDependency
import org.mulesoft.apb.project.internal.parser.{APBEnv, DescriptorParser}

import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.util.{Failure, Success}

object ProjectBuilder {
  def apply(dependencyFetcher: DependencyFetcher, loaders: List[ResourceLoader]): ProjectBuilder = {
    new ProjectBuilder(dependencyFetcher, loaders)
  }
}

class ProjectBuilder(dependencyFetcher: DependencyFetcher,
                     loaders: List[ResourceLoader],
                     cacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder)
    extends PlatformSecrets {

  type GavPath = String

  private val provider: String => DescriptorParser                  = APBEnv.getProvider
  private val cache: mutable.Map[GavPath, Future[ParsedDependency]] = mutable.Map.empty
  private val gavLoaders: mutable.Map[GavPath, ResourceLoader]      = mutable.Map.empty

  private def parseDescriptor(c: Content): ProjectDescriptor =
    provider(c.stream.toString).parse()

  def buildFromDirectory(directory: String): Future[ProjectConfiguration] = {
    val eventualContent = fetchDescriptorFile(directory)
    eventualContent.flatMap(c => buildFromContent(c.stream.toString, Some(directory)))
  }

  def buildFromContent(descriptorAsString: String, directory: Option[String]): Future[ProjectConfiguration] =
    build(APBEnv.getParser(descriptorAsString).parse(directory))

  def build(project: ProjectDescriptor): Future[ProjectConfiguration] = buildProjectConfig(project, Set.empty)

  private def fetchDescriptorFile(directory: String) = {
    val config = AMFGraphConfiguration.empty().withResourceLoaders(loaders)
    platform.fetchContent(directory + "/" + APBEnv.descriptorFileName, config)
  }

  def processDependency(dependency: ProjectDependency, dependenciesInPath: Set[Gav]): Future[ParsedDependency] = {
    if (dependenciesInPath.contains(dependency.gav))
      throw CyclicProjectDependencyException(dependency.gav, dependenciesInPath)
    else if (cache.contains(dependency.gav.path)) cache(dependency.gav.path)
    else if (assetIsUnreachable(dependency)) {
      Future.successful(UnreachableDependency(dependency))
    } else {
      val loader = dependencyFetcher.wrapFetch(dependency.gav.groupId,
                                               dependency.gav.assetId,
                                               dependency.gav.version,
                                               APBEnv.projectProtocol)
      gavLoaders.put(dependency.gav.path, loader)
      loader.fetch(APBEnv.projectProtocol + APBEnv.descriptorFileName).transformWith {
        case Success(content) =>
          val parsedDependency = parseDependency(dependency, content, dependenciesInPath, loader)
          cache.put(dependency.gav.path, parsedDependency)
          parsedDependency
        case Failure(exception) =>
          exception match {
            case UnreachableGavException(_) => Future.successful(internal.dependency.UnreachableDependency(dependency))
            case _                          => throw exception
          }
      }
    }
  }

  private def parseDependency(dependency: ProjectDependency,
                              content: Content,
                              dependenciesInPath: Set[Gav],
                              loader: ResourceLoader) = {
    val descriptor = parseDescriptor(content)
    val parsedDependency = buildProjectConfig(descriptor, dependenciesInPath + descriptor.gav)
      .flatMap { config =>
        config.webApiParseConfig
          .withResourceLoader(loader)
          .baseUnitClient()
          .parse(APBEnv.projectProtocol + descriptor.main)
          .map { r =>
            buildParsedDependency(r.baseUnit,
                                  config.errors.addTreeErrors(r.results.toList),
                                  descriptor,
                                  dependency.scope)
          }
      }
    parsedDependency
  }

  private def assetIsUnreachable(dependency: ProjectDependency) = {
    val Gav(group, asset, version) = dependency.gav
    !dependencyFetcher.accepts(group, asset, version)
  }

  private def buildParsedDependency(bu: BaseUnit,
                                    errors: ProjectErrors,
                                    projectDescriptor: ProjectDescriptor,
                                    scope: DependencyScope): ParsedDependency = {
    scope match {
      case ExtensionScope if bu.isInstanceOf[Dialect] =>
        ExtensionDependency(bu.asInstanceOf[Dialect], projectDescriptor, errors)
      case ValidationScope if bu.isInstanceOf[DialectInstance] =>
        ProfileDependency(bu.asInstanceOf[DialectInstance], projectDescriptor, errors)
      case _ => DesignDependency(bu, projectDescriptor, errors)

    }
  }

  def buildProjectConfig(descriptor: ProjectDescriptor, seenDependencies: Set[Gav]): Future[ProjectConfiguration] = {
    val parsed = descriptor.dependencies.map(dep => processDependency(dep, seenDependencies))

    Future.sequence(parsed).flatMap { seq =>
      val design: ListBuffer[DesignDependency]       = ListBuffer.empty
      val profile: ListBuffer[ProfileDependency]     = ListBuffer.empty
      val extension: ListBuffer[ExtensionDependency] = ListBuffer.empty
      seq.foreach {
        case d: DesignDependency    => design += d
        case p: ProfileDependency   => profile += p
        case e: ExtensionDependency => extension += e
        case _                      => // ignore
      }
      val errors = seq.foldLeft(ProjectErrors()) { (acc, curr) =>
        acc.add(curr.errors)
      }
      ValidationProfileDialectLoader.dialect.map { d =>
        new ProjectConfiguration(design.toList,
                                 profile.toList,
                                 extension.toList,
                                 gavLoaders.toMap,
                                 descriptor,
                                 d,
                                 cacheBuilder,
                                 loaders,
                                 errors)
      }
    }
  }
}
