package org.mulesoft.apb.project.internal.parser

import org.mulesoft.apb.project.client.scala.model._
import org.yaml.model.{YDocument, YMap}
import org.yaml.parser.JsonParser

trait ParserProviderCompanion {
  def getProvider: String => DescriptorParser
  val descriptorFileName: String
  val projectProtocol: String
}

abstract class DescriptorParser(val content: String) {
  def populate(): Unit
  val builder = new DescriptorBuilder()

  def parse(): ProjectDescriptor = {
    populate()
    builder.build()
  }

  def parse(base: Option[String]): ProjectDescriptor = {
    base.foreach(builder.withBase)
    parse()
  }
}

class ExchangeJsonParser(override val content: String) extends DescriptorParser(content) {
  val ast: YDocument = JsonParser(content)().document()
  val map: YMap      = ast.node.as[YMap]

  override def populate(): Unit = {
    map.stringValueFor("main", builder.withMain)
    map.stringValueFor("name", builder.withName)
    map.stringValueFor("classifier", builder.withClassifier)

    val (group, asset, version) = parseGav(map)
    builder.withGroupId(group)
    builder.withAssetId(asset)
    builder.withVersion(version)

    map.key("tags", e => builder.withTags(e.value.as[Seq[String]]))
    map.key("dependencies", e => parseDependencies(e.value.as[Seq[YMap]]))
  }

  private def parseDependencies(dependencies: Seq[YMap]) = {
    val result = dependencies.map { d =>
      val scope                   = d.stringValueFor("scope").map(e => DependencyScope(e)).getOrElse(DesignScope)
      val (group, asset, version) = parseGav(d)
      ProjectDependency(scope, Gav(group, asset, version))
    }
    builder.withDependencies(result)
  }

  private def parseGav(m: YMap): (String, String, String) = {
    (m.stringValueFor("groupId").getOrElse(""),
     m.stringValueFor("assetId").getOrElse(""),
     m.stringValueFor("version").getOrElse(""))
  }
}

object ExchangeJsonParserCompanion extends ParserProviderCompanion {
  def apply(content: String) = new ExchangeJsonParser(content)

  override def getProvider: String => DescriptorParser = apply

  override val descriptorFileName: String = "exchange.json"
  override val projectProtocol: String    = "file:///"
}
