package org.mulesoft.apb.project.internal.engine

import amf.core.client.common.validation.SeverityLevels
import amf.core.client.scala.validation.AMFValidationResult
import org.mulesoft.apb.project.client.scala.ProjectErrors
import org.mulesoft.apb.project.client.scala.dependency.{
  Dependency,
  ExtensionDependency,
  ProfileDependency,
  ProvidedDependency
}
import org.mulesoft.apb.project.client.scala.model.{ExtensionScope, Gav, ProjectDependency}
import org.mulesoft.apb.project.internal.validations.ProjectValidations.SuggestedExtensionDependency

import scala.collection.mutable.ListBuffer

object SuggestedExtensionAdditions {

  def apply(project: Gav,
            profileDeps: ListBuffer[ProfileDependency],
            extensionDeps: ListBuffer[ExtensionDependency]): ProjectErrors =
    detectValidatableExtensions(project, profileDeps, extensionDeps)

  private def detectValidatableExtensions(project: Gav,
                                          profileDeps: ListBuffer[ProfileDependency],
                                          extensionDeps: ListBuffer[ExtensionDependency]): ProjectErrors = {
    val notPresentExtensionsByProfile = buildSuggestedExtensionByProfile(profileDeps, extensionDeps)
    val results                       = createErrorResults(project, notPresentExtensionsByProfile)
    ProjectErrors().addProjectErrors(results)
  }

  private def createErrorResults(project: Gav, notPresentExtensionsByProfile: Map[Gav, Set[Gav]]) = {
    notPresentExtensionsByProfile.flatMap {
      case (profileGav, extensionGavs) =>
        extensionGavs.map(extensionGav => suggestion(project, profileGav, extensionGav))
    }.toList
  }

  private def buildSuggestedExtensionByProfile(profile: ListBuffer[ProfileDependency],
                                               extension: ListBuffer[ExtensionDependency]) = {
    val projectExtensionGavs          = extension.map(_.descriptor.gav).toSet
    val extensionsUsedByEachProfile   = profile.map(p => p.descriptor.gav -> getProvidedExtensions(p)).toMap
    val notPresentExtensionsByProfile = extensionsUsedByEachProfile.mapValues(x => x.diff(projectExtensionGavs))
    notPresentExtensionsByProfile
  }

  private def suggestedExtensionMessage(profileGav: Gav, extensionGav: Gav) = {
    s"Design-extension dependency: ${extensionGav.path} should be added when using validation dependency: ${profileGav.path}"
  }

  private def filterExtensionDependencies(dependency: Dependency): Seq[ProjectDependency] = {
    dependency.descriptor.dependencies.filter(p => p.scope == ExtensionScope)
  }

  private def getProvidedExtensions(dependency: Dependency) =
    filterExtensionDependencies(dependency).filter(x => x.shouldBeProvided).map(_.gav).toSet

  private def suggestion(project: Gav, profile: Gav, extension: Gav): AMFValidationResult = {
    AMFValidationResult(
      suggestedExtensionMessage(profile, extension),
      SeverityLevels.WARNING,
      "",
      None,
      SuggestedExtensionDependency.id,
      None,
      Some(project.path),
      ProvidedDependency(profile, extension)
    )
  }
}
