package org.mulesoft.apb.project.internal.parser

import amf.aml.client.scala.model.document.DialectInstance
import amf.aml.client.scala.model.domain.DialectDomainElement
import amf.core.client.scala.vocabulary.Namespace
import org.mulesoft.apb.project.client.scala.model._

object AMLExchangeDescriptorAdapter {

  private val Main              = (Namespace.Data + "main").iri()
  private val Classifier        = (Namespace.Data + "classifier").iri()
  private val Name              = (Namespace.Data + "name").iri()
  private val GroupId           = (Namespace.Data + "groupId").iri()
  private val Version           = (Namespace.Data + "version").iri()
  private val AssetId           = (Namespace.Data + "assetId").iri()
  private val Dependencies      = (Namespace.Data + "dependencies").iri()
  private val Tags              = (Namespace.Data + "tags").iri()
  private val Scope             = (Namespace.Data + "scope").iri()
  private val Provided          = (Namespace.Data + "provided").iri()

  def adapt(instance: DialectInstance, base: String, descriptorVersion: String): ProjectDescriptor = {
    val encodes      = instance.encodes.asInstanceOf[DialectDomainElement]
    val main         = string(encodes, Main)
    val fullMain     = if (base.nonEmpty) base + "/" + main else main
    val classifier   = option(encodes, Classifier, _.toString)
    val name         = string(encodes, Name)
    val groupId      = string(encodes, GroupId)
    val assetId      = string(encodes, AssetId)
    val version      = string(encodes, Version)
    val tags         = array(encodes, Tags, _.toString)
    val dependencies = obj(encodes, Dependencies, adaptDependency)
    ProjectDescriptor(fullMain,
                      name,
                      descriptorVersion,
                      Gav(groupId, assetId, version),
                      classifier,
                      tags,
                      dependencies)
  }

  private def array[T](node: DialectDomainElement, iri: String, transform: Any => T): Seq[T] = {
    node.getScalarByProperty(iri).map(transform)
  }

  private def string(node: DialectDomainElement, iri: String): String = {
    node.getScalarByProperty(iri).headOption.map(_.toString).getOrElse("")
  }

  private def boolean(node: DialectDomainElement, iri: String): Boolean = {
    node.getScalarByProperty(iri).headOption.exists {
      case bool: Boolean => bool
      case _ => false
    }
  }

  private def option[T](node: DialectDomainElement, iri: String, transform: Any => T): Option[T] = {
    node.getScalarByProperty(iri).map(transform).headOption
  }

  private def obj[T](node: DialectDomainElement, iri: String, transform: DialectDomainElement => T): Seq[T] = {
    node.getObjectByProperty(iri).map(transform)
  }

  private def adaptDependency(element: DialectDomainElement): ProjectDependency = {
    val groupId          = string(element, GroupId)
    val assetId          = string(element, AssetId)
    val version          = string(element, Version)
    val scope            = option(element, Scope, _.toString).map(DependencyScope(_)).getOrElse(DesignScope)
    val hasExplicitScope = option(element, Scope, _.toString).isDefined
    val provided = boolean(element, Provided)
    ProjectDependency(scope, Gav(groupId, assetId, version), hasExplicitScope, provided)
  }
}
