package org.mulesoft.apb.project.client.scala

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.apicontract.client.scala._
import amf.core.client.scala.config.UnitCache
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.{AMFValidationReport, AMFValidationResult}
import amf.core.internal.remote.Spec
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptor
import org.mulesoft.apb.project.internal.config.ClassifierConfigProvider
import org.mulesoft.apb.project.internal.dependency.APBResourceLoader

case class ProjectErrors(treeErrors: List[AMFValidationResult] = Nil, projectErrors: List[AMFValidationResult] = Nil) {
  private[apb] def addTreeErrors(errors: List[AMFValidationResult]) = copy(treeErrors = treeErrors ++ errors)
  private[apb] def addProjectError(error: AMFValidationResult)      = copy(projectErrors = error :: projectErrors)
  private[apb] def addProjectErrors(report: AMFValidationReport) =
    copy(projectErrors = report.results.toList ::: projectErrors)
  private[apb] def addProjectErrors(errors: List[AMFValidationResult]) = copy(projectErrors = errors ::: projectErrors)
  private[apb] def add(errors: ProjectErrors): ProjectErrors =
    copy(treeErrors = this.treeErrors ++ errors.treeErrors, projectErrors = this.projectErrors ++ errors.projectErrors)
  private[apb] def allErrors: List[AMFValidationResult] = treeErrors ::: projectErrors
}

case class ProjectConfiguration(designDependencies: Seq[DesignDependency],
                                profileDependencies: Seq[ProfileDependency],
                                extensionDependencies: Seq[ExtensionDependency],
                                dependenciesLoaders: Map[String, ResourceLoader],
                                descriptor: ProjectDescriptor,
                                validationDialect: Dialect,
                                unitCacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder,
                                envLoaders: List[ResourceLoader],
                                errors: ProjectErrors) {

  private val mainLoader: ResourceLoader = new APBResourceLoader(dependenciesLoaders)
  private lazy val unitsCache: UnitCache = unitCacheBuilder.build(designDependencies)

  protected val classifierConfig: AMFConfiguration = descriptor.classifier
    .map(ClassifierConfigProvider.get)
    .getOrElse(APIConfiguration.API())

  def mainFile: String                       = descriptor.main
  def allDependencies: Seq[ParsedDependency] = designDependencies ++ extensionDependencies ++ profileDependencies
  def getDialectFor(instance: DialectInstance): Option[Dialect] =
    webApiParseConfig.configurationState().findDialectFor(instance)
  def webApiParseConfig: AMFConfiguration     = dependenciesConfig(WebAPIConfiguration.WebAPI())
  def projectParseConfig: AMFConfiguration    = dependenciesConfig(classifierConfig)
  def configFor(spec: Spec): AMFConfiguration = dependenciesConfig(WebAPIConfiguration.fromSpec(spec))

  def dependenciesConfig(base: AMFConfiguration): AMFConfiguration = {
    val inc =
      base.withResourceLoaders(List(mainLoader) ++ envLoaders).withDialect(validationDialect).withUnitCache(unitsCache)
    extensionDependencies
      .foldLeft(inc) { (acc, curr) =>
        acc.withDialect(curr.dialect)
      }
  }
}
