package org.mulesoft.apb.project.internal.engine

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationResult
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model._
import org.mulesoft.apb.project.client.scala.{ProjectConfiguration, ProjectErrors}
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.internal.parser.APBEnv

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

case class AnypointTreeBuilder(dependencyFetcher: DependencyFetcher,
                               loaders: List[ResourceLoader],
                               cacheBuilder: UnitCacheBuilder)
    extends TreeBuilderTemplate(dependencyFetcher, loaders, cacheBuilder) {

  protected def parseDependency(descriptor: DescriptorParseResult,
                                dependency: ProjectDependency,
                                dependenciesInPath: Set[Gav],
                                loader: ResourceLoader): Future[ParsedDependency] = {
    val DescriptorParseResult(project, report) = descriptor
    for {
      config   <- buildProjectConfig(project, dependenciesInPath + project.gav)
      mainFile <- parseMainFile(loader, project, config)
    } yield {
      val projectErrors = setDependencyGavAsLocation(dependency, report.results.toList)
      val errors = config.errors.addTreeErrors(mainFile.results.toList).addProjectErrors(projectErrors)
      buildParsedDependency(mainFile.baseUnit, errors, project, dependency.scope)
    }
  }

  private def parseMainFile(loader: ResourceLoader,
                            descriptor: ProjectDescriptor,
                            config: ProjectConfiguration) = {
    config.webApiParseConfig
      .withResourceLoader(loader)
      .baseUnitClient()
      .parse(APBEnv.projectProtocol + descriptor.main)
  }

  private def buildParsedDependency(bu: BaseUnit,
                                    errors: ProjectErrors,
                                    projectDescriptor: ProjectDescriptor,
                                    scope: DependencyScope): ParsedDependency = {
    scope match {
      case ExtensionScope if bu.isInstanceOf[Dialect] =>
        ExtensionDependency(bu.asInstanceOf[Dialect], projectDescriptor, errors)
      case ValidationScope if bu.isInstanceOf[DialectInstance] =>
        ProfileDependency(bu.asInstanceOf[DialectInstance], projectDescriptor, errors)
      case _ => DesignDependency(bu, projectDescriptor, errors)

    }
  }

  protected def setDependencyGavAsLocation(dependency: ProjectDependency, errors: List[AMFValidationResult]): List[AMFValidationResult] = {
    val location = DescriptorErrorLocation(dependency)
    errors.map(_.copy(location = Some(location)))
  }
}
