package org.mulesoft.apb.project.internal.engine

import amf.core.client.common.validation.SeverityLevels
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationResult
import org.mulesoft.apb.project.client.scala.dependency.{Dependency, NotParsedDependency, NothingUnitCacheBuilder}
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model.{DesignScope, Gav, ProjectDependency, ProjectDescriptor}
import org.mulesoft.apb.project.client.scala.ProjectErrors
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.internal.validations.ProjectValidations.ScopePresentInLegacyProject

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

case class LegacyTreeBuilder(dependencyFetcher: DependencyFetcher, loaders: List[ResourceLoader])
    extends TreeBuilderTemplate(dependencyFetcher, loaders, NothingUnitCacheBuilder) {
  override protected def parseDependency(descriptor: DescriptorParseResult,
                                         dependency: ProjectDependency,
                                         dependenciesInPath: Set[Gav],
                                         loader: ResourceLoader): Future[Dependency] = {
    val DescriptorParseResult(project, report) = descriptor
    val processedDependencies = project.dependencies
      .map(dep => processDependency(project, dep, dependenciesInPath + project.gav))

    // we don't care about the dependencies, we only care the get processed so we can use them in resource loader
    Future.sequence(processedDependencies).map { _ =>
      val isRoot = dependenciesInPath.isEmpty
      val errors = scopeErrors(project, dependency, isRoot = isRoot).addProjectErrors(report.results.toList)
      NotParsedDependency(project, errors)
    }
  }

  private def scopeErrors(descriptor: ProjectDescriptor,
                          dependency: ProjectDependency,
                          errors: ProjectErrors = ProjectErrors(),
                          isRoot: Boolean): ProjectErrors = {
    if (dependency.hasExplicitScope && dependency.scope != DesignScope) {
      errors.addProjectError(scopeError(descriptor, dependency, isRoot = isRoot))
    } else errors
  }

  private def scopeError(descriptor: ProjectDescriptor, dependency: ProjectDependency, isRoot: Boolean) = {
    val location = if (isRoot) DescriptorErrorLocation.forRoot() else DescriptorErrorLocation(dependency)
    AMFValidationResult(
      s"Dependency ${dependency.gav.path} in module ${descriptor.gav.path} shouldn't have a defined scope in a 0.1.0 project",
      SeverityLevels.WARNING,
      dependency.gav.path,
      None,
      ScopePresentInLegacyProject.id,
      None,
      Some(location),
      Unit
    )
  }
}
