package org.mulesoft.apb.project.internal.listener

import amf.core.client.scala.errorhandling.AMFErrorHandler
import amf.core.client.scala.parse.document.Reference
import org.mulesoft.apb.project.client.scala.dependency.{DesignDependency, ExchangeModulePathHandler}
import org.mulesoft.apb.project.internal.validations.ProjectValidations.MissingCompanionLibUses

/**
  * Listener to detect if a list of companion libs is being used from the api.
  * If not, an error will be raised.
  */
class IncludedCompanionLibReferenceListener private (companionLibs: Seq[DesignDependency])
    extends RawReferenceListener {
  override protected def checkReferences(references: Seq[Reference], location: String, eh: AMFErrorHandler): Unit = {
    val requiredLibs      = companionLibs.flatMap(l => normalizeLocation(l.location))
    val currentReferences = references.flatMap(r => normalizeLocation(r.url))

    requiredLibs.filter(rl => !currentReferences.contains(rl)).foreach { missed =>
      eh.violation(
        MissingCompanionLibUses,
        location,
        MissingCompanionLibUses.message + s" Add library ${missed}"
      )
    }

  }

  private def normalizeLocation(location: String): Option[String] =
    ExchangeModulePathHandler(location).map(_.exchangePath)
}

object IncludedCompanionLibReferenceListener {
  def apply(companionLibs: Seq[DesignDependency]): Option[IncludedCompanionLibReferenceListener] = {
    if (companionLibs.nonEmpty) Some(new IncludedCompanionLibReferenceListener(companionLibs))
    else None
  }
}
