package org.mulesoft.apb.project.internal.parser

import amf.aml.client.scala.model.document.DialectInstance
import amf.aml.client.scala.model.domain.DialectDomainElement
import amf.core.client.scala.vocabulary.Namespace
import org.mulesoft.apb.project.client.scala.model._

object AMLExchangeDescriptorAdapter {

  private val Main                  = (Namespace.Data + "main").iri()
  private val Classifier            = (Namespace.Data + "classifier").iri()
  private val Name                  = (Namespace.Data + "name").iri()
  private val GroupId               = (Namespace.Data + "groupId").iri()
  private val Version               = (Namespace.Data + "version").iri()
  private val ApiVersion            = (Namespace.Data + "apiVersion").iri()
  private val OrganizationId        = (Namespace.Data + "organizationId").iri()
  private val OriginalFormatVersion = (Namespace.Data + "originalFormatVersion").iri()
  private val Metadata              = (Namespace.Data + "metadata").iri()
  private val BranchId              = (Namespace.Data + "branchId").iri()
  private val CommitId              = (Namespace.Data + "commitId").iri()
  private val ProjectId             = (Namespace.Data + "projectId").iri()
  private val BackwardsCompatible   = (Namespace.Data + "backwardsCompatible").iri()

  private val AssetId      = (Namespace.Data + "assetId").iri()
  private val Dependencies = (Namespace.Data + "dependencies").iri()
  private val Tags         = (Namespace.Data + "tags").iri()
  private val Scope        = (Namespace.Data + "scope").iri()
  private val Provided     = (Namespace.Data + "provided").iri()

  def adapt(instance: DialectInstance, base: String, descriptorVersion: String): ProjectDescriptor = {
    val encodes                           = instance.encodes.asInstanceOf[DialectDomainElement]
    val main                              = string(encodes, Main)
    val fullMain                          = if (base.nonEmpty) base + "/" + main else main
    val classifier                        = stringOption(encodes, Classifier)
    val name                              = string(encodes, Name)
    val groupId                           = string(encodes, GroupId)
    val assetId                           = string(encodes, AssetId)
    val version                           = string(encodes, Version)
    val apiVersion                        = stringOption(encodes, ApiVersion)
    val organizationId                    = stringOption(encodes, OrganizationId)
    val originalFormatVersion             = stringOption(encodes, OriginalFormatVersion)
    val backwardsCompatible               = boolOption(encodes, BackwardsCompatible)
    val metadata: Option[MetadataElement] = obj(encodes, Metadata, adaptMetadata).headOption
    val tags                              = array(encodes, Tags, _.toString)
    val dependencies                      = obj(encodes, Dependencies, adaptDependency(classifier))
    ProjectDescriptor(
      fullMain,
      name,
      descriptorVersion,
      Gav(groupId, assetId, version),
      classifier,
      tags,
      apiVersion,
      organizationId,
      originalFormatVersion,
      metadata,
      backwardsCompatible,
      dependencies
    )
  }

  private def array[T](node: DialectDomainElement, iri: String, transform: Any => T): Seq[T] = {
    node.getScalarByProperty(iri).map(transform)
  }

  private def string(node: DialectDomainElement, iri: String): String = {
    node.getScalarByProperty(iri).headOption.map(_.toString).getOrElse("")
  }

  private def boolean(node: DialectDomainElement, iri: String): Boolean = {
    node.getScalarByProperty(iri).headOption.exists {
      case bool: Boolean => bool
      case _             => false
    }
  }

  private def stringOption(node: DialectDomainElement, iri: String): Option[String] = {
    option(node, iri, _.toString)
  }

  private def boolOption(node: DialectDomainElement, iri: String): Option[Boolean] = {
    option(node, iri, {
      case b: Boolean => b
      case _          => false
    })
  }

  private def option[T](node: DialectDomainElement, iri: String, transform: Any => T): Option[T] = {
    node.getScalarByProperty(iri).map(transform).headOption
  }

  private def obj[T](node: DialectDomainElement, iri: String, transform: DialectDomainElement => T): Seq[T] = {
    node.getObjectByProperty(iri).map(transform)
  }

  private def adaptDependency(parentClassifier: Option[String])(element: DialectDomainElement): ProjectDependency = {
    val groupId          = string(element, GroupId)
    val assetId          = string(element, AssetId)
    val version          = string(element, Version)
    val scope            = stringOption(element, Scope).map(DependencyScope(_)).getOrElse(DesignScope)
    val hasExplicitScope = stringOption(element, Scope).isDefined
    val provided         = boolean(element, Provided)
    ProjectDependency(scope, Gav(groupId, assetId, version), hasExplicitScope, provided, parentClassifier)
  }

  private def adaptMetadata(element: DialectDomainElement): MetadataElement = {
    val branchId  = stringOption(element, BranchId)
    val commitId  = stringOption(element, CommitId)
    val projectId = stringOption(element, ProjectId)
    MetadataElement(branchId, commitId, projectId)
  }
}
