package org.mulesoft.apb.project.client.scala.descriptor

import amf.aml.client.scala.AMLConfiguration
import amf.aml.client.scala.model.document.DialectInstance
import amf.core.client.scala.errorhandling.IgnoringErrorHandler
import amf.core.client.scala.parse.document.SyamlParsedDocument
import amf.core.internal.errorhandling.SYAMLJsonErrorHandler
import amf.core.internal.plugins.syntax.SyamlSyntaxRenderPlugin
import amf.core.internal.remote.Mimes.`application/json`
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptor
import org.mulesoft.apb.project.internal.generated.AnypointDescriptorDialectLoader
import org.mulesoft.apb.project.internal.parser.{AMLDescriptorPlugin, ParserProviderCompanion, YMapLocalOps}
import org.yaml.model.{YDocument, YMap}
import org.yaml.parser.JsonParser

import java.io.StringWriter
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

object ExchangeDescriptorHandler extends ParserProviderCompanion {
  def apply() = new ExchangeDescriptorHandler("")

  def apply(base: Option[String]) = new ExchangeDescriptorHandler(base.getOrElse(""))

  override def getProvider: Option[String] => DescriptorParser = apply

  override val descriptorFileName: String = "exchange.json"
  override val projectProtocol: String    = "file:///"
}

case class ExchangeDescriptorHandler private[apb] (base: String) extends DescriptorParser with DescriptorRender {

  def parse(content: String): Future[DescriptorParseResult] = {
    for {
      version <- parseDescriptorVersion(content)
      dialect <- AnypointDescriptorDialectLoader.dialect(version)
      config <- Future.successful(
        AMLConfiguration.predefined().withDialect(dialect).withPlugin(AMLDescriptorPlugin(dialect))
      )
      parsed <- config.baseUnitClient().parseContent(content, `application/json`)
      report <- config.baseUnitClient().validate(parsed.baseUnit.cloneUnit())
    } yield {
      val finalReport = report.copy(results = parsed.results ++ report.results)
      val descriptor  = ProjectDescriptor(parsed.baseUnit.asInstanceOf[DialectInstance], base, version)
      DescriptorParseResult(descriptor, finalReport)
    }
  }

  def render(descriptor: ProjectDescriptor): Future[String] = {
    for {
      dialect <- AnypointDescriptorDialectLoader.dialect(descriptor.descriptorVersion)
      config  <- Future.successful(AMLConfiguration.predefined().withDialect(dialect))
    } yield {
      val renderedAst =
        config.elementClient().renderElement(descriptor.instance.encodes, descriptor.instance.references)
      val renderedString =
        SyamlSyntaxRenderPlugin
          .emit(`application/json`, SyamlParsedDocument(document = YDocument(renderedAst)), new StringWriter)
          .map(_.toString)
          .getOrElse("")
      renderedString
    }
  }

  private def parseDescriptorVersion(content: String): Future[String] = {
    implicit val eh: SYAMLJsonErrorHandler = SYAMLJsonErrorHandler(IgnoringErrorHandler)
    val json                               = JsonParser(content)
    val rawValue                           = json.document().node.as[YMap].stringValueFor("descriptorVersion")
    val result = rawValue match {
      case Some("1.0.0") => "1.0.0"
      case Some("0.1.0") => "0.1.0"
      case Some(_)       => "1.0.0"
      case _             => "0.1.0"
    }
    Future.successful(result)
  }
}
