package org.mulesoft.apb.project.internal.engine

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.core.client.scala.AMFParseResult
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationResult
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model._
import org.mulesoft.apb.project.client.scala.{ProjectConfiguration, ProjectErrors}

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

case class AnypointTreeBuilder(dependencyFetcher: DependencyFetcher,
                               loaders: List[ResourceLoader],
                               cacheBuilder: UnitCacheBuilder)
    extends TreeBuilderTemplate(dependencyFetcher, loaders, cacheBuilder)
    with CollisionComputationHelper {

  override def build(project: ProjectDescriptor): Future[ProjectConfiguration] =
    super
      .build(project)
      .flatMap { lookupRisksForRoot(_, loaders) }
      .map { assignMigrationRisks }

  private def processDependency(descriptorParse: DescriptorParseResult,
                                dependency: ProjectDependency,
                                dependenciesInPath: Set[Gav],
                                loader: ResourceLoader): Future[ParsedDependency] = {
    val DescriptorParseResult(descriptor, report) = descriptorParse
    for {
      config       <- buildProjectConfig(descriptor, dependenciesInPath + descriptor.gav)
      mainFileInfo <- MainFileInfoBuilder.build(loader, descriptor, config, dependency.parentClassifier)
      parsedDependency <- {
        val descriptorErrors = setDependencyGavAsLocation(dependency, report.results.toList)
        val errors           = computeNextErrors(config.errors, mainFileInfo.result, descriptorErrors)
        buildParsedDependency(mainFileInfo, errors, dependency.scope, config.migrationRisks)
      }
    } yield parsedDependency
  }

  protected override def parseDependency(descriptorParse: DescriptorParseResult,
                                         dependency: ProjectDependency,
                                         dependenciesInPath: Set[Gav],
                                         loader: ResourceLoader): Future[ParsedDependency] = {
    for {
      dependency       <- processDependency(descriptorParse, dependency, dependenciesInPath, loader)
      mergedDependency <- MigrationRisk.compute(dependency, loader)
    } yield {
      mergedDependency
    }
  }

  private def computeNextErrors(errors: ProjectErrors,
                                mainFile: AMFParseResult,
                                descriptorErrors: List[AMFValidationResult]) = {
    errors
      .addTreeErrors(mainFile.results.toList)
      .addProjectErrors(descriptorErrors)
  }

  private def buildParsedDependency(mainFileInfo: MainFileInfo,
                                    errors: ProjectErrors,
                                    scope: DependencyScope,
                                    migrationRisks: MigrationRisks): Future[ParsedDependency] = {
    scope match {
      case ExtensionScope if mainFileInfo.baseUnit.isInstanceOf[Dialect] =>
        mainFileInfo.parseCompanionLibrary.map { companion =>
          companion.foreach(t => t._2.withReferences(Seq(mainFileInfo.baseUnit)))
          ExtensionDependency(mainFileInfo.baseUnit.asInstanceOf[Dialect], companion, mainFileInfo.descriptor, errors)
        }
      case ValidationScope if mainFileInfo.baseUnit.isInstanceOf[DialectInstance] =>
        Future.successful(
          ProfileDependency(mainFileInfo.baseUnit.asInstanceOf[DialectInstance], mainFileInfo.descriptor, errors))
      case _ =>
        Future.successful(
          DesignDependency(mainFileInfo.baseUnit,
                           mainFileInfo.descriptor,
                           errors,
                           mainFileInfo.config.absolutePathListener,
                           migrationRisks))
    }
  }

  protected def setDependencyGavAsLocation(dependency: ProjectDependency,
                                           errors: List[AMFValidationResult]): List[AMFValidationResult] = {
    val location = DescriptorErrorLocation(dependency)
    errors.map(_.copy(location = Some(location)))
  }
}
