package org.mulesoft.apb.project.internal.engine

import amf.apicontract.client.scala.APIConfiguration
import amf.core.client.common.validation.SeverityLevels
import amf.core.client.scala.AMFParseResult
import amf.core.client.scala.model.document.{BaseUnit, Module}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationResult
import amf.core.internal.remote.Spec
import org.mulesoft.apb.project.client.scala.ProjectConfiguration
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptor
import org.mulesoft.apb.project.internal.dependency.UnreachableDependency
import org.mulesoft.apb.project.internal.parser.APBEnv
import org.mulesoft.apb.project.internal.validations.ProjectValidations.UnreacheableAsset

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

case class MainFileInfo(
    loader: ResourceLoader,
    descriptor: ProjectDescriptor,
    config: ProjectConfiguration,
    result: AMFParseResult,
    parentClassifier: Option[String]
) {
  val baseUnit: BaseUnit = result.baseUnit

  def parseCompanionLibrary: Future[Option[(String, Module)]] = {
    parentClassifier.flatMap {
      case v if v.contains("raml") => Some((Spec.RAML10, ".raml"))
//      case Spec.GRAPHQL => ".graphql"
//      case Spec.GRPC => ".proto3"
      case _ => Some((Spec.RAML10, ".raml")) // todo: current semex classifier is aml? need previous classifier?
    } map { case (spec, extension) =>
      val companionName = descriptor.main.substring(0, descriptor.main.lastIndexOf(".")) + extension
      parseCompanion(spec, companionName).recoverWith { case _: Throwable => Future.successful(None) }
    } getOrElse (Future.successful(None))
  }

  private def parseCompanion(spec: Spec, companionName: String) = {
    APIConfiguration
      .fromSpec(spec) // todo: check for graphql and grpc later on
      .withResourceLoader(loader)
      .baseUnitClient()
      .parseLibrary(companionName)
      .map(l => Some((companionName, l.library)))
  }

}

object MainFileInfoBuilder {
  def build(
      loader: ResourceLoader,
      descriptor: ProjectDescriptor,
      config: ProjectConfiguration,
      parentClassifier: Option[String]
  ): Future[MainFileInfo] = {

    def notFoundResult(e: Throwable) =
      AMFValidationResult(
        e.getMessage,
        SeverityLevels.VIOLATION,
        "",
        None,
        UnreacheableAsset.id,
        None,
        Some(descriptor.gav.path + APBEnv.descriptorFileName),
        Unit
      )

    val amfConfig = config.webApiParseConfig
      .withResourceLoader(loader)
    // need to enforce listeners regarding project configuration value at AnypointTreeBuilder. Root descriptor version rules how to check paths.
    val withListeners = config.getAnypointConfigListeners.foldLeft(amfConfig) { (acc, curr) =>
      acc.withEventListener(curr)
    }

    withListeners
      .baseUnitClient()
      .parse(APBEnv.projectProtocol + descriptor.main)
      .recoverWith({ case e: Throwable =>
        Future.successful(new AMFParseResult(UnreachableDependency.emptyUnit, Seq(notFoundResult(e))))
      })
      .map(MainFileInfo(loader, descriptor, config, _, parentClassifier))
  }

}
