package org.mulesoft.apb.project.internal.parser

import amf.aml.client.scala.model.domain.DialectDomainElement
import org.mulesoft.apb.project.client.scala.model._

object AMLExchangeDescriptorAdapter {

  def array[T](node: DialectDomainElement, iri: String, transform: Any => T): Seq[T] = {
    node.getScalarByProperty(iri).map(transform)
  }

  def string(node: DialectDomainElement, iri: String): String = {
    node.getScalarByProperty(iri).headOption.map(_.toString).getOrElse("")
  }

  def boolean(node: DialectDomainElement, iri: String): Boolean = {
    node.getScalarByProperty(iri).headOption.exists {
      case bool: Boolean => bool
      case _             => false
    }
  }

  def stringOption(node: DialectDomainElement, iri: String): Option[String] = {
    option(node, iri, _.toString)
  }

  def boolOption(node: DialectDomainElement, iri: String): Option[Boolean] = {
    option(
      node,
      iri,
      {
        case b: Boolean => b
        case _          => false
      }
    )
  }

  def option[T](node: DialectDomainElement, iri: String, transform: Any => T): Option[T] = {
    node.getScalarByProperty(iri).map(transform).headOption
  }

  def obj[T](node: DialectDomainElement, iri: String, transform: DialectDomainElement => T): Seq[T] = {
    node.getObjectByProperty(iri).map(transform)
  }

  def adaptDependency(parentClassifier: Option[String])(element: DialectDomainElement): ProjectDependency = {
    val groupId = string(element, ProjectDescriptorModel.GroupId)
    val assetId = string(element, ProjectDescriptorModel.AssetId)
    val version = string(element, ProjectDescriptorModel.Version)
    val scope   = stringOption(element, ProjectDescriptorModel.Scope).map(DependencyScope(_)).getOrElse(DesignScope)
    val hasExplicitScope = stringOption(element, ProjectDescriptorModel.Scope).isDefined
    val provided         = boolean(element, ProjectDescriptorModel.Provided)
    ProjectDependency(scope, Gav(groupId, assetId, version), hasExplicitScope, provided, parentClassifier)
  }

  def adaptMetadata(element: DialectDomainElement): MetadataElement = {
    val branchId  = stringOption(element, ProjectDescriptorModel.BranchId)
    val commitId  = stringOption(element, ProjectDescriptorModel.CommitId)
    val projectId = stringOption(element, ProjectDescriptorModel.ProjectId)
    MetadataElement(branchId, commitId, projectId)
  }
}
