package org.mulesoft.apb.project.client.scala

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.apicontract.client.scala._
import amf.core.client.scala.config.UnitCache
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.remote.Spec
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptor
import org.mulesoft.apb.project.client.scala.transformation.APBRaml10EditingPipeline
import org.mulesoft.apb.project.internal.config.ClassifierConfigProvider
import org.mulesoft.apb.project.internal.dependency.APBResourceLoader
import org.mulesoft.apb.project.internal.engine.MigrationRisks
import org.mulesoft.apb.project.internal.listener.{AbsolutePathDetectionRawReferenceListener, RawReferenceListener}

case class ProjectConfiguration(
    designDependencies: Seq[DesignDependency],
    profileDependencies: Seq[ProfileDependency],
    extensionDependencies: Seq[ExtensionDependency],
    dependenciesLoaders: Map[String, ResourceLoader],
    descriptor: ProjectDescriptor,
    validationDialect: Dialect,
    unitCacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder,
    envLoaders: List[ResourceLoader],
    errors: ProjectErrors,
    private[apb] val migrationRisks: MigrationRisks = MigrationRisks.empty
) {

  implicit class AMFConfigPopulator(base: AMFConfiguration) {
    def addExtensions(): AMFConfiguration = extensionDependencies.foldLeft(base) { (acc, curr) =>
      acc.withDialect(curr.dialect)
    }

    def addListeners(): AMFConfiguration = getListeners.foldLeft(base) { (acc, curr) =>
      acc.withEventListener(curr)
    }
  }

  val mainLoader: ResourceLoader = new APBResourceLoader(dependenciesLoaders)
  private lazy val companionLibDependencies: Seq[DesignDependency] =
    extensionDependencies.flatMap(_.companionLibDependency)
  lazy val unitsCache: UnitCache =
    unitCacheBuilder.build(designDependencies ++ companionLibDependencies)

  protected val classifierConfig: AMFConfiguration = descriptor.classifier
    .map(ClassifierConfigProvider.get)
    .getOrElse(APIConfiguration.API())

  lazy val absolutePathListener: Option[AbsolutePathDetectionRawReferenceListener] =
    AbsolutePathDetectionRawReferenceListener(descriptor)

  val getLegacyConfigListeners: Iterable[RawReferenceListener] = Nil

  def getAnypointConfigListeners: Iterable[RawReferenceListener] = absolutePathListener

  def getListeners: Iterable[RawReferenceListener] =
    if (!descriptor.isLegacyDescriptor) getAnypointConfigListeners else getLegacyConfigListeners

  def mainFile: String = descriptor.main

  def allDependencies: Seq[ParsedDependency] = designDependencies ++ extensionDependencies ++ profileDependencies

  def getDialectFor(instance: DialectInstance): Option[Dialect] =
    webApiParseConfig.configurationState().findDialectFor(instance)

  def webApiParseConfig: AMFConfiguration = dependenciesConfig(WebAPIConfiguration.WebAPI())

  def projectParseConfig: AMFConfiguration = dependenciesConfig(classifierConfig)

  def configFor(spec: Spec): AMFConfiguration = dependenciesConfig(WebAPIConfiguration.fromSpec(spec))

  def dependenciesConfig(base: AMFConfiguration): AMFConfiguration = addCompanionLibCheckStage(
      base
        .withResourceLoaders(List(mainLoader) ++ envLoaders)
        .withDialect(validationDialect)
        .withUnitCache(unitsCache)
        .addListeners()
        .addExtensions()
  )

  private def addCompanionLibCheckStage(base: AMFConfiguration): AMFConfiguration = {
    if (descriptor.classifierSpec.contains(Spec.RAML10))
      base.withTransformationPipeline(new APBRaml10EditingPipeline(companionLibDependencies))
    else base
  }
}
