package org.mulesoft.apb.project.client.scala.model

import amf.aml.client.scala.model.document.DialectInstance
import amf.aml.client.scala.model.domain.DialectDomainElement
import amf.core.internal.remote.Spec
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptorModel._
import org.mulesoft.apb.project.internal.generated.AnypointDescriptorDialectLoader
import org.mulesoft.apb.project.internal.parser.AMLExchangeDescriptorAdapter._

import scala.scalajs.js.annotation.{JSExport, JSExportAll, JSExportTopLevel}

case class ProjectDescriptor(instance: DialectInstance, base: String, descriptorVersion: String) extends GavAware {

  private val encodes = instance.encodes.asInstanceOf[DialectDomainElement]

  val main: String                          = getMain
  val name: String                          = string(encodes, Name)
  val gav: Gav                              = getGav
  val classifier: Option[String]            = stringOption(encodes, Classifier)
  val tags: Seq[String]                     = array(encodes, Tags, _.toString)
  val apiVersion: Option[String]            = stringOption(encodes, ApiVersion)
  val organizationId: Option[String]        = stringOption(encodes, OrganizationId)
  val originalFormatVersion: Option[String] = stringOption(encodes, OriginalFormatVersion)
  val metadata: Option[MetadataElement]     = obj(encodes, Metadata, adaptMetadata).headOption
  val backwardsCompatible: Option[Boolean]  = boolOption(encodes, BackwardsCompatible)
  val publishWithRefFiles: Option[Boolean]  = boolOption(encodes, PublishWithRefFiles)
  val description: Option[String]           = stringOption(encodes, Description)
  val projectId: Option[String]             = stringOption(encodes, ProjectId)
  val dependencies: Seq[ProjectDependency]  = obj(encodes, Dependencies, adaptDependency(classifier))

  private def getMain: String = {
    val main: String = string(encodes, Main)
    if (base.nonEmpty) base + "/" + main else main
  }

  private def getGav: Gav = {
    val groupId: String = string(encodes, GroupId)
    val assetId: String = string(encodes, AssetId)
    val version: String = string(encodes, Version)
    Gav(groupId, assetId, version)
  }

  def isLegacyDescriptor: Boolean = descriptorVersion == AnypointDescriptorDialectLoader.`VERSION_0.1.0`

  def classifierSpec: Option[Spec] = {
    classifier match {
      case Some(v) if v.contains("raml")                         => Some(Spec.RAML10)
      case Some(v) if v.contains("oas") || v.contains("swagger") => Some(Spec.OAS20)
      case Some(v) if v.contains("openapi")                      => Some(Spec.OAS30)
      case Some(v) if v.contains("graphql")                      => Some(Spec.GRAPHQL)
      case Some(v) if v.contains("grpc")                         => Some(Spec.GRPC)
      case Some(v) if v.contains("asyncapi")                     => Some(Spec.ASYNC20)
      case _                                                     => None
    }
  }

  protected[apb] def copyWithLiteralProperties(
      properties: Seq[LiteralProperty],
      base: Option[String] = None
  ): ProjectDescriptor = {
    val clonedInstance = instance.cloneUnit().asInstanceOf[DialectInstance]
    val clonedEncodes  = clonedInstance.encodes.asInstanceOf[DialectDomainElement]
    properties.foreach {
      case LiteralProperty(iri, stringValue: String)   => clonedEncodes.withLiteralProperty(iri, stringValue)
      case LiteralProperty(iri, booleanValue: Boolean) => clonedEncodes.withLiteralProperty(iri, booleanValue)
      case LiteralProperty(iri, seqValue: Seq[Any])    => clonedEncodes.withLiteralProperty(iri, seqValue.toList)
      case _ => // Ignored. In case of add other type fields we need to add more cases here
    }
    this.copy(clonedInstance, base.getOrElse(this.base), descriptorVersion)
  }

}

protected[apb] case class LiteralProperty(iri: String, value: Any)

object Gav {
  def unapply(gav: String): Option[Gav] = gav.split("/") match {
    case Array(group, asset, version) => Some(Gav(group, asset, version))
    case _                            => None
  }
}

case class Gav(groupId: String, assetId: String, version: String) {
  val path: String = s"$groupId/$assetId/$version/"

  def equalsTo(groupId: String, assetId: String, version: String): Boolean =
    this.groupId == groupId && this.assetId == assetId && this.version == version

  def nonEmpty: Boolean = this.groupId.nonEmpty || this.assetId.nonEmpty || this.version.nonEmpty
}

case class MetadataElement(branchId: Option[String], commitId: Option[String], projectId: Option[String])

case class ProjectDependency(
    scope: DependencyScope,
    gav: Gav,
    hasExplicitScope: Boolean,
    shouldBeProvided: Boolean,
    parentClassifier: Option[String]
) extends GavAware {

  val absolutePath: String = "/exchange_modules/" + gav.path + "exchange.json"
}

trait GavAware {
  def gav: Gav
}

@JSExportAll
sealed case class DependencyScope(scope: String)

@JSExportAll
@JSExportTopLevel("DependencyScope")
object DependencyScope {

  @JSExport("apply")
  def apply(scope: String): DependencyScope = scope match {
    case ValidationScope.scope => ValidationScope
    case ExtensionScope.scope  => ExtensionScope
    case _                     => DesignScope
  }

  val Design: DependencyScope     = DesignScope
  val Validation: DependencyScope = ValidationScope
  val Extension: DependencyScope  = ExtensionScope
}

@JSExportAll
@JSExportTopLevel("DesignScope")
object DesignScope extends DependencyScope("design")

@JSExportAll
@JSExportTopLevel("ValidationScope")
object ValidationScope extends DependencyScope("validation")

@JSExportAll
@JSExportTopLevel("ExtensionScope")
object ExtensionScope extends DependencyScope("design-extension")
