package org.mulesoft.apb.project.internal.listener

import amf.core.client.scala.errorhandling.AMFErrorHandler
import amf.core.client.scala.parse.document.Reference
import org.mulesoft.apb.project.client.scala.dependency.ExchangeModulePathHandler
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptor
import org.mulesoft.apb.project.internal.validations.ProjectValidations.AbsolutePathUsage
import org.mulesoft.common.client.lexical.SourceLocation

import scala.collection.mutable

/** Listener to detect there is any absolute path being used. This should be used only working with api fragments
  * classifiers If any absolute path inclusion is present, an error will be raised.
  */
class AbsolutePathDetectionRawReferenceListener private () extends RawReferenceListener {

  private val absoluteReferences = mutable.ArrayBuffer[Reference]()

  override protected def checkReferences(references: Seq[Reference], location: String, eh: AMFErrorHandler): Unit = {
    val absolutes = references.filter(isAbsolute)
    absolutes.foreach(abs => processReference(abs, location, eh))

    absoluteReferences.appendAll(absolutes)
  }

  private def processReference(reference: Reference, location: String, eh: AMFErrorHandler) = {
    reference.refs.map { r =>
      val range = r.reduceToLocation()
      eh.violation(
          AbsolutePathUsage,
          reference.url,
          s"Detected an absolute file path '${reference.url}'. Paths starting with '/' are no longer allowed in fragments.",
          SourceLocation(location, range.start.line, range.start.column, range.end.line, range.end.column)
      )
    }
  }

  private def isAbsolute(ref: Reference) = {
    ref.url.startsWith("/") && !ref.url.contains(ExchangeModulePathHandler.EXCHANGE_MODULES)
  }

  def getDetectedAbsoluteRefs: List[Reference] = absoluteReferences.toList
  def getDetectedAbsoluteRefUri: Set[String]   = absoluteReferences.map(_.url).toSet
}

object AbsolutePathDetectionRawReferenceListener {
  def apply(projectDescriptor: ProjectDescriptor): Option[AbsolutePathDetectionRawReferenceListener] = {
    projectDescriptor.classifier match {
      case Some(v) if v.contains("fragment") && v.contains("raml") =>
        Some(new AbsolutePathDetectionRawReferenceListener())
      case _ => None
    }
  }
}
