package org.mulesoft.apb.project.internal.engine

import amf.core.client.common.remote.Content
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.unsafe.PlatformSecrets
import amf.custom.validation.internal.report.loaders.ProfileDialectLoader
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.descriptor.{DescriptorParseResult, DescriptorParser}
import org.mulesoft.apb.project.client.scala.environment.{DependencyFetcher, UnreachableGavException}
import org.mulesoft.apb.project.client.scala.model.{Gav, GavAware, ProjectDependency, ProjectDescriptor}
import org.mulesoft.apb.project.client.scala.{ProjectConfiguration, ProjectErrors}
import org.mulesoft.apb.project.internal.dependency.UnreachableDependency
import org.mulesoft.apb.project.internal.parser.APBEnv

import scala.collection.mutable
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.util.{Failure, Success}

abstract class TreeBuilderTemplate(
    dependencyFetcher: DependencyFetcher,
    loaders: List[ResourceLoader],
    cacheBuilder: UnitCacheBuilder
) extends ProjectEngine
    with PlatformSecrets {

  type GavPath = String

  protected val cache: mutable.Map[GavPath, Future[Dependency]]  = mutable.Map.empty
  protected val gavLoaders: mutable.Map[GavPath, ResourceLoader] = mutable.Map.empty

  private val provider: Option[String] => DescriptorParser = APBEnv.getProvider

  protected def parseDescriptor(content: Content): Future[DescriptorParseResult] =
    provider(None).parse(content.stream.toString)

  override def build(project: ProjectDescriptor): Future[ProjectConfiguration] =
    buildProjectConfig(project, Set.empty)

  protected def buildProjectConfig(
      descriptor: ProjectDescriptor,
      seenDependencies: Set[Gav]
  ): Future[ProjectConfiguration] = {
    val parsed        = descriptor.dependencies.map(dep => processDependency(descriptor, dep, seenDependencies))
    val isRootProject = seenDependencies.isEmpty

    for {
      deps           <- Future.sequence(parsed)
      profileDialect <- ProfileDialectLoader.dialect
    } yield {
      val parsedDependencies           = deps.collect({ case p: ParsedDependency => p })
      val (design, profile, extension) = PartitionedDependencies(parsedDependencies)
      var errors                       = dependencyErrors(deps)
      if (isRootProject) errors = errors.add(SuggestedExtensionAdditions(descriptor.gav, parsedDependencies))
      val risks = deps.collect { case c: DesignDependency => c }.foldLeft(MigrationRisks.empty) { (acc, curr) =>
        acc.add(curr.risks)
      }

      ProjectConfiguration(
          design,
          profile,
          extension,
          gavLoaders.toMap,
          descriptor,
          profileDialect,
          cacheBuilder,
          loaders,
          errors,
          risks
      )
    }
  }

  private def dependencyErrors(seq: Seq[Dependency]) = {
    seq.foldLeft(ProjectErrors()) { (acc, curr) =>
      acc.add(curr.errors)
    }
  }

  private def assetIsUnreachable(dependency: ProjectDependency) = {
    val Gav(group, asset, version) = dependency.gav
    !dependencyFetcher.accepts(group, asset, version, dependency.classifier, dependency.packaging)
  }

  protected def processDependency(
      descriptor: ProjectDescriptor,
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav]
  ): Future[Dependency] = {
    if (dependenciesInPath.contains(dependency.gav))
      throw CyclicProjectDependencyException(dependency.gav, dependenciesInPath)
    else if (cache.contains(dependency.gav.path)) cache(dependency.gav.path)
    else if (assetIsUnreachable(dependency)) Future.successful(UnreachableDependency(descriptor, dependency))
    else {
      val loader = cachedFetchLoader(dependency)
      loader.fetch(APBEnv.projectProtocol + APBEnv.descriptorFileName).transformWith {
        case Success(content)   => memoizeDependency(dependency, dependenciesInPath, loader, content)
        case Failure(exception) => onFetchFailure(descriptor, dependency, exception)
      }
    }
  }

  private def onFetchFailure(descriptor: ProjectDescriptor, dependency: ProjectDependency, exception: Throwable) = {
    exception match {
      case UnreachableGavException(_) => Future.successful(UnreachableDependency(descriptor, dependency))
      case _                          => throw exception
    }
  }

  private def memoizeDependency(
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader,
      content: Content
  ): Future[Dependency] = {
    val parsedDependency = parseDependency(dependency, content, dependenciesInPath, loader)
    cache.put(dependency.gav.path, parsedDependency)
    parsedDependency
  }

  private def fetchLoader(dependency: ProjectDependency) = {
    dependencyFetcher.wrapFetch(
        dependency.gav.groupId,
        dependency.gav.assetId,
        dependency.gav.version,
        APBEnv.projectProtocol,
        dependency.classifier,
        dependency.packaging
    )
  }

  protected def cachedFetchLoader(dependency: ProjectDependency): ResourceLoader = {
    val loader = fetchLoader(dependency)
    gavLoaders.put(dependency.gav.path, loader)
    loader
  }

  private def parseDependency(
      dependency: ProjectDependency,
      content: Content,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader
  ): Future[Dependency] = {
    parseDescriptor(content).flatMap { parsed =>
      parseDependency(parsed, dependency, dependenciesInPath, loader)
    }
  }

  protected def parseDependency(
      descriptor: DescriptorParseResult,
      dependency: ProjectDependency,
      dependenciesInPath: Set[Gav],
      loader: ResourceLoader
  ): Future[Dependency]

  protected def errorLocation(gav: Gav) = s"exchange_modules/${gav.path}exchange.json"
}
