package org.mulesoft.apb.project.client.scala.descriptor

import amf.aml.client.scala.model.document.{Dialect, DialectInstance, DialectInstanceProcessingData}
import amf.aml.client.scala.model.domain.{DialectDomainElement, NodeMapping}
import amf.core.internal.parser.domain.Annotations
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptorModel._
import org.mulesoft.apb.project.client.scala.model.{Gav, MetadataElement, ProjectDependency, ProjectDescriptor}
import org.mulesoft.apb.project.internal.generated.AnypointDescriptorDialectLoader._

import java.util.UUID
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

case class ExchangeDescriptorBuilder private[apb] (private var main: String) {

  private var descriptorVersion: String             = `VERSION_0.1.0`
  private var name: String                          = ""
  private var gav: Gav                              = Gav("", "", "")
  private var classifier: Option[String]            = None
  private var tags: Seq[String]                     = Nil
  private var apiVersion: Option[String]            = None
  private var organizationId: Option[String]        = None
  private var originalFormatVersion: Option[String] = None
  private var metadata: Option[MetadataElement]     = None
  private var backwardsCompatible: Option[Boolean]  = None
  private var publishWithRefFiles: Option[Boolean]  = None
  private var description: Option[String]           = None
  private var projectId: Option[String]             = None
  private var dependencies: Seq[ProjectDependency]  = Nil

  def withMain(main: String): ExchangeDescriptorBuilder = {
    this.main = main
    this
  }
  def withDescriptorVersion(descriptorVersion: String): ExchangeDescriptorBuilder = {
    this.descriptorVersion = descriptorVersion
    this
  }
  def withName(name: String): ExchangeDescriptorBuilder = {
    this.name = name
    this
  }
  def withGav(gav: Gav): ExchangeDescriptorBuilder = {
    this.gav = gav
    this
  }
  def withGav(groupId: String, assetId: String, version: String): ExchangeDescriptorBuilder = {
    this.gav = Gav(groupId, assetId, version)
    this
  }
  def withClassifier(classifier: String): ExchangeDescriptorBuilder = {
    this.classifier = Some(classifier)
    this
  }
  def withTags(tags: Seq[String]): ExchangeDescriptorBuilder = {
    this.tags = tags
    this
  }
  def withTag(tag: String): ExchangeDescriptorBuilder = {
    this.tags = this.tags :+ tag
    this
  }
  def withApiVersion(apiVersion: String): ExchangeDescriptorBuilder = {
    this.apiVersion = Some(apiVersion)
    this
  }
  def withOrganizationId(organizationId: String): ExchangeDescriptorBuilder = {
    this.organizationId = Some(organizationId)
    this
  }
  def withOriginalFormatVersion(originalFormatVersion: String): ExchangeDescriptorBuilder = {
    this.originalFormatVersion = Some(originalFormatVersion)
    this
  }
  def withMetadata(metadata: MetadataElement): ExchangeDescriptorBuilder = {
    this.metadata = Some(metadata)
    this
  }
  def withMetadata(
      branchId: Option[String],
      commitId: Option[String],
      projectId: Option[String]
  ): ExchangeDescriptorBuilder = {
    this.metadata = Some(MetadataElement(branchId, commitId, projectId))
    this
  }
  def withBackwardsCompatible(backwardsCompatible: Boolean): ExchangeDescriptorBuilder = {
    this.backwardsCompatible = Some(backwardsCompatible)
    this
  }
  def withPublishWithRefFiles(publishWithRefFiles: Boolean): ExchangeDescriptorBuilder = {
    this.publishWithRefFiles = Some(publishWithRefFiles)
    this
  }
  def withDescription(description: String): ExchangeDescriptorBuilder = {
    this.description = Some(description)
    this
  }
  def withProjectId(projectId: String): ExchangeDescriptorBuilder = {
    this.projectId = Some(projectId)
    this
  }
  def withDependencies(dependencies: Seq[ProjectDependency]): ExchangeDescriptorBuilder = {
    this.dependencies = dependencies
    this
  }
  def withDependency(dependency: ProjectDependency): ExchangeDescriptorBuilder = {
    this.dependencies = this.dependencies :+ dependency
    this
  }

  def build(): Future[ProjectDescriptor] = {
    for {
      dialect <- dialect(descriptorVersion)
    } yield {
      val mapping = getNodeMappingId(dialect, "Descriptor")
      val encodes = DialectDomainElement(randomId, mapping, Annotations())
      val dialectInstance = DialectInstance()
        .withLocation(main)
        .withId(main)
        .withProcessingData(DialectInstanceProcessingData().withTransformed(false).withDefinedBy(dialect.id))
        .withDefinedBy(dialect.id)
        .withEncodes(encodes)
      setFields(encodes, dialect)
      ProjectDescriptor(dialectInstance, "", descriptorVersion)
    }
  }

  private def isLegacyDescriptor: Boolean = descriptorVersion == `VERSION_0.1.0`

  private def setFields(domainElement: DialectDomainElement, dialect: Dialect): Unit = {
    domainElement.withLiteralProperty(Main, main)
    classifier.foreach(c => domainElement.withLiteralProperty(Classifier, c))
    apiVersion.foreach(av => domainElement.withLiteralProperty(ApiVersion, av))
    organizationId.foreach(oid => domainElement.withLiteralProperty(OrganizationId, oid))
    originalFormatVersion.foreach(ofv => domainElement.withLiteralProperty(OriginalFormatVersion, ofv))
    backwardsCompatible.foreach(bc => domainElement.withLiteralProperty(BackwardsCompatible, bc))
    publishWithRefFiles.foreach(pwrf => domainElement.withLiteralProperty(PublishWithRefFiles, pwrf))
    description.foreach(d => domainElement.withLiteralProperty(Description, d))
    projectId.foreach(pid => domainElement.withLiteralProperty(ProjectId, pid))
    metadata.foreach(m => domainElement.withObjectProperty(Metadata, generateMetadataDomainElement(m, dialect)))
    if (descriptorVersion.nonEmpty) domainElement.withLiteralProperty(DescriptorVersion, descriptorVersion)
    if (name.nonEmpty) domainElement.withLiteralProperty(Name, name)
    if (tags.nonEmpty) domainElement.withLiteralProperty(Tags, tags.toList)
    if (gav.nonEmpty) populateGav(gav, domainElement)
    if (dependencies.nonEmpty)
      domainElement.withObjectCollectionProperty(
          Dependencies,
          dependencies.map(d => generateProjectDependencyDomainElement(d, dialect))
      )
  }

  private def generateMetadataDomainElement(metadata: MetadataElement, dialect: Dialect): DialectDomainElement = {
    val mapping         = getNodeMappingId(dialect, "MetadataNode")
    val metadataElement = DialectDomainElement(randomId, mapping, Annotations())
    metadata.projectId.foreach(pid => metadataElement.withLiteralProperty(ProjectId, pid))
    metadata.branchId.foreach(bid => metadataElement.withLiteralProperty(BranchId, bid))
    metadata.commitId.foreach(cid => metadataElement.withLiteralProperty(CommitId, cid))
    metadataElement
  }

  private def generateProjectDependencyDomainElement(
      dependency: ProjectDependency,
      dialect: Dialect
  ): DialectDomainElement = {
    val mapping           = getNodeMappingId(dialect, "Dependency")
    val dependencyElement = DialectDomainElement(randomId, mapping, Annotations())
    if (dependency.gav.nonEmpty) populateGav(dependency.gav, dependencyElement)
    if (dependency.hasExplicitScope) dependencyElement.withLiteralProperty(Scope, dependency.scope.scope)
    dependency.packaging.foreach(packaging => dependencyElement.withLiteralProperty(Packaging, packaging))
    dependency.classifier.foreach(classifier => dependencyElement.withLiteralProperty(Classifier, classifier))
    if (!isLegacyDescriptor) dependencyElement.withLiteralProperty(Provided, dependency.shouldBeProvided)
    dependencyElement
  }

  private def getNodeMappingId(dialect: Dialect, nodeName: String): NodeMapping =
    dialect.declares
      .collect { case nodeMapping: NodeMapping =>
        nodeMapping
      }
      .find(declaration => declaration.name.value() == nodeName)
      .get

  private def populateGav(gav: Gav, domainElement: DialectDomainElement): Unit = {
    domainElement.withLiteralProperty(GroupId, gav.groupId)
    domainElement.withLiteralProperty(AssetId, gav.assetId)
    domainElement.withLiteralProperty(Version, gav.version)
  }

  private def randomId: String = UUID.randomUUID().toString
}

object ExchangeDescriptorBuilder {

  def apply(main: String) = new ExchangeDescriptorBuilder(main)

  def apply(baseDescriptor: ProjectDescriptor): ExchangeDescriptorBuilder = {
    val builder = ExchangeDescriptorBuilder(baseDescriptor.main)

    builder.withDescriptorVersion(baseDescriptor.descriptorVersion)
    builder.withName(baseDescriptor.name)
    builder.withGav(baseDescriptor.gav)
    baseDescriptor.classifier.foreach(builder.withClassifier)
    builder.withTags(baseDescriptor.tags)
    baseDescriptor.apiVersion.foreach(builder.withApiVersion)
    baseDescriptor.organizationId.foreach(builder.withOrganizationId)
    baseDescriptor.originalFormatVersion.foreach(builder.withOriginalFormatVersion)
    baseDescriptor.metadata.foreach(builder.withMetadata)
    baseDescriptor.backwardsCompatible.foreach(builder.withBackwardsCompatible)
    baseDescriptor.publishWithRefFiles.foreach(builder.withPublishWithRefFiles)
    baseDescriptor.description.foreach(builder.withDescription)
    baseDescriptor.projectId.foreach(builder.withProjectId)
    builder.withDependencies(baseDescriptor.dependencies)
  }
}
