package org.mulesoft.apb.project.client.scala.dependency

import amf.core.client.scala.config.{CachedReference, UnitCache}
import amf.core.client.scala.model.document.BaseUnit
import org.mulesoft.apb.project.internal.dependency.ExchangeModulePathHandler

import scala.concurrent.Future

trait UnitCacheBuilder {
  def build(deps: Seq[DesignDependency]): UnitCache
}

/** This implementation of UnitCacheBuilder that clones the baseUnit of the provided dependencies and creates a
  * DirectUnitCache. This implementation allows that BaseUnits of dependencies are not shared by several
  * BaseUnitClients.
  */
object CloneOnceUnitCacheBuilder extends UnitCacheBuilder {
  override def build(dependencies: Seq[DesignDependency]): UnitCache = {
    val dependencyMap = dependencies
      .map(dependency => ExchangeModulePathHandler.buildPath(dependency.descriptor) -> dependency.baseUnit.cloneUnit())
      .toMap

    DirectUnitCache(dependencyMap)
  }
}

/** This implementation of UnitCache returns the instances of BaseUnit that that composite. It should be used in when
  * parsing a single "Unit" because the elements in the cache are mutable.
  *
  * @param baseUnits
  *   a Map whit the BaseUnits to return given a URL.
  */
case class DirectUnitCache(private val baseUnits: Map[String, BaseUnit]) extends UnitCache {
  override def fetch(url: String): Future[CachedReference] = {
    val dependencyPath = ExchangeModulePathHandler(url).map(_.exchangePath).getOrElse(url)
    Future.successful(CachedReference(dependencyPath, baseUnits(dependencyPath)))
  }
}
