package org.mulesoft.apb.project.internal.transformation

import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.errorhandling.AMFErrorHandler
import amf.core.client.scala.model.document.{BaseUnit, FieldsFilter, Module}
import amf.core.client.scala.model.domain.extensions.CustomDomainProperty
import amf.core.client.scala.model.domain.{AmfElement, NamedDomainElement}
import amf.core.client.scala.transform.TransformationStep
import amf.core.internal.annotations.DeclaredElement
import org.mulesoft.apb.project.client.scala.dependency.DesignDependency
import org.mulesoft.apb.project.internal.dependency.ExchangeModulePathHandler
import org.mulesoft.apb.project.internal.dependency.ExchangeModulePathHandler.EXCHANGE_MODULES
import org.mulesoft.apb.project.internal.validations.ProjectValidations.MissingCompanionLibUses

class CompanionLibCheckTransformationStage(companionLibs: Seq[DesignDependency]) extends TransformationStep() {

  private val companionExtensionNames: Seq[String] = getCompanionSemanticExtensionsNames
  private val companionLibLocations: Seq[String]   = companionLibs.flatMap(l => normalizeLocation(l.location()))

  private def getCompanionSemanticExtensionsNames: Seq[String] = companionLibs
    .map(_.baseUnit)
    .collect { case library: Module =>
      library.declares.collect { case annotationDeclaration: CustomDomainProperty =>
        annotationDeclaration.name.toString()
      }
    }
    .flatten

  override def transform(
      model: BaseUnit,
      errorHandler: AMFErrorHandler,
      configuration: AMFGraphConfiguration
  ): BaseUnit = {
    model.iterator(fieldsFilter = FieldsFilter.All).foreach {
      case annotationDeclaration: CustomDomainProperty if applies(annotationDeclaration) =>
        companionLibLocations.foreach(missed =>
          errorHandler.violation(
              MissingCompanionLibUses,
              annotationDeclaration,
              MissingCompanionLibUses.message + s" Import library $missed and reference that extensions",
              annotations = annotationDeclaration.annotations
          )
        )
      case _ => // Ignore
    }
    model
  }

  private def applies(annotationDeclaration: CustomDomainProperty): Boolean =
    notExchangeDependency(annotationDeclaration) &&
      isDeclaration(annotationDeclaration) &&
      isSemanticExtensionName(annotationDeclaration)

  private def notExchangeDependency(element: AmfElement): Boolean =
    !element.location().exists(loc => loc.contains(EXCHANGE_MODULES))
  private def isDeclaration(element: AmfElement): Boolean = element.annotations.contains(classOf[DeclaredElement])
  private def isSemanticExtensionName(element: NamedDomainElement) =
    companionExtensionNames.contains(element.name.toString())

  private def normalizeLocation(location: String): Option[String] =
    ExchangeModulePathHandler(location).map(_.exchangePath.stripPrefix("file://"))

}
