package org.mulesoft.apb.project.client.scala.model

import amf.core.client.scala.model.{BoolField, StrField}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.domain.metamodel.jsonldschema.JsonLDEntityModel
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.apb.project.client.scala.model.ProjectDependencyModel.{AssetId, GroupId, Version}
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptorModel.{Metadata => MetadataIri, _}
import org.mulesoft.apb.project.internal.generated.DescriptorSchemaLoader.LegacyVersion

import scala.language.implicitConversions

case class ProjectDescriptor private[apb] (override private[apb] val wrapped: JsonLDObject, base: String)
    extends JsonLDObjectWrapper
    with GavAware {

  override type This = ProjectDescriptor
  implicit def metadataConversion(obj: JsonLDObject): Option[Metadata]               = Option(obj).map(Metadata.apply)
  implicit def dependencyConversion(objs: Seq[JsonLDObject]): Seq[ProjectDependency] = objs.map(ProjectDependency.apply)
  implicit def instanceConversion(objs: Seq[JsonLDObject]): Seq[Instance]            = objs.map(Instance.apply)

  lazy val main: String                          = getMain
  lazy val name: String                          = get[StrField](ProjectDescriptorModel.Name)
  lazy val descriptorVersion: String             = getDescriptorVersion
  lazy val gav: Gav                              = getGav
  lazy val classifier: Option[String]            = get[StrField](Classifier)
  lazy val tags: Seq[String]                     = get[Seq[StrField]](Tags)
  lazy val apiVersion: Option[String]            = get[StrField](ApiVersion)
  lazy val organizationId: Option[String]        = get[StrField](OrganizationId)
  lazy val originalFormatVersion: Option[String] = get[StrField](OriginalFormatVersion)
  lazy val metadata: Option[Metadata]            = get[JsonLDObject](MetadataIri)
  lazy val backwardsCompatible: Option[Boolean]  = get[BoolField](BackwardsCompatible)
  lazy val publishWithRefFiles: Option[Boolean]  = get[BoolField](PublishWithRefFiles)
  lazy val description: Option[String]           = get[StrField](Description)
  lazy val projectId: Option[String]             = get[StrField](ProjectId)
  lazy val dependencies: Seq[ProjectDependency]  = get[Seq[JsonLDObject]](Dependencies)
  lazy val instances: Seq[Instance]              = get[Seq[JsonLDObject]](ApiInstances)

  private def getMain: String = {
    val main: String = get[StrField](ProjectDescriptorModel.Main)
    if (base.nonEmpty) base + "/" + main else main
  }

  private def getDescriptorVersion: String = {
    val version: StrField = get(DescriptorVersion)
    version.option() match {
      case Some("1.0.0")       => "1.0.0"
      case Some(LegacyVersion) => LegacyVersion
      case Some(_)             => "1.0.0"
      case None                => LegacyVersion
    }
  }

  override protected def copy(newWrapped: JsonLDObject): This = ProjectDescriptor(newWrapped, base)
  private[apb] def isRaml: Boolean                            = classifier.contains("raml")
  private def getGav: Gav = {
    val groupId: String = get[StrField](GroupId)
    val assetId: String = get[StrField](AssetId)
    val version: String = get[StrField](Version)
    Gav(groupId, assetId, version)
  }

  def isLegacyDescriptor: Boolean = descriptorVersion == LegacyVersion

  def withMain(main: String): ProjectDescriptor = {
    val next = wrapped.withProperty(ProjectDescriptorModel.Main, main)
    ProjectDescriptor(next, base)
  }
  def withDependency(dependency: ProjectDependency): ProjectDescriptor = {
    val newDependencies = dependencies.filterNot(_.gav == dependency.gav) :+ dependency

    copy(wrapped.withObjPropertyCollection(Dependencies, newDependencies.map(_.wrapped)))
  }

  def withInstance(instance: Instance): ProjectDescriptor = {
    val newInstances = instances.filterNot(_.gcl == instance.gcl) :+ instance

    withInstances(newInstances)
  }

  def withInstances(instances: Seq[Instance]): ProjectDescriptor = {
    copy(wrapped.withObjPropertyCollection(ApiInstances, instances.map(_.wrapped)))
  }

  def withName(name: String): ProjectDescriptor                 = copy(wrapped.withProperty(Name, name))
  def withDescriptorVersion(version: String): ProjectDescriptor = copy(wrapped.withProperty(DescriptorVersion, version))
  def withGav(gav: Gav): ProjectDescriptor = {
    val updated = wrapped
      .withProperty(GroupId, gav.groupId)
      .withProperty(AssetId, gav.assetId)
      .withProperty(Version, gav.version)
    copy(updated)
  }
  def withClassifier(classifier: String): ProjectDescriptor = copy(wrapped.withProperty(Classifier, classifier))
  def withTags(tags: Seq[String]): ProjectDescriptor        = copy(wrapped.withStringPropertyCollection(Tags, tags))
  def withTag(tag: String): ProjectDescriptor               = withTags(tags :+ tag)
  def withApiVersion(apiVersion: String): ProjectDescriptor = copy(wrapped.withProperty(ApiVersion, apiVersion))
  def withOrganizationId(organizationId: String): ProjectDescriptor = copy(
      wrapped.withProperty(OrganizationId, organizationId)
  )
  def withOriginalFormatVersion(originalFormatVersion: String): ProjectDescriptor = copy(
      wrapped.withProperty(OriginalFormatVersion, originalFormatVersion)
  )
  def withMetadata(metadata: Metadata): ProjectDescriptor = copy(wrapped.withProperty(MetadataIri, metadata.wrapped))
  def withBackwardsCompatible(value: Boolean): ProjectDescriptor = copy(
      wrapped.withProperty(BackwardsCompatible, value)
  )
  def withPublishWithRefFiles(value: Boolean): ProjectDescriptor = copy(
      wrapped.withProperty(PublishWithRefFiles, value)
  )
  def withDescription(description: String): ProjectDescriptor = copy(wrapped.withProperty(Description, description))
  def withProjectId(projectId: String): ProjectDescriptor     = copy(wrapped.withProperty(ProjectId, projectId))
}

object ProjectDescriptor {
  def apply(main: String, base: String): ProjectDescriptor = {
    val dObject = JsonLDObject.empty(ProjectDescriptorModel.entityModel, JsonPath.empty).set(Main, main)
    ProjectDescriptor(dObject, base)
  }

  def apply(main: String): ProjectDescriptor = apply(main, "")
}
