package org.mulesoft.apb.project.client.scala.dependency

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.core.client.scala.model.document.{BaseUnit, Module}
import amf.core.client.scala.vocabulary.Namespace
import org.mulesoft.apb.project.client.scala.ProjectErrors
import org.mulesoft.apb.project.client.scala.model._
import org.mulesoft.apb.project.internal.engine.MigrationRisks
import org.mulesoft.apb.project.internal.listener.AbsolutePathDetectionRawReferenceListener

trait Dependency {
  val descriptor: ProjectDescriptor
  val errors: ProjectErrors
}

case class NotParsedDependency(descriptor: ProjectDescriptor, errors: ProjectErrors) extends Dependency

abstract class ParsedDependency(val baseUnit: BaseUnit, val descriptor: ProjectDescriptor, val errors: ProjectErrors)
    extends Dependency {
  val scope: DependencyScope
  val location: String = baseUnit.location().getOrElse(baseUnit.id)
}

case class DesignDependency(
    override val baseUnit: BaseUnit,
    override val descriptor: ProjectDescriptor,
    override val errors: ProjectErrors,
    val listener: Option[AbsolutePathDetectionRawReferenceListener],
    val risks: MigrationRisks
) extends ParsedDependency(baseUnit, descriptor, errors) {
  override val scope: DependencyScope = DesignScope

  def update(migrationRisks: MigrationRisks): DesignDependency =
    this.copy(risks = migrationRisks)
}

case class ExtensionDependency(
    dialect: Dialect,
    companionLib: Option[(String, Module)],
    override val descriptor: ProjectDescriptor,
    override val errors: ProjectErrors
) extends ParsedDependency(dialect, descriptor, errors) {
  override val scope: DependencyScope = ExtensionScope

  val companionLibDependency: Option[DesignDependency] =
    companionLib.map { case (name, library) =>
      DesignDependency(
          library,
          descriptor.withMain(name),
          errors,
          None,
          MigrationRisks.empty
      )
    }
}

case class ProfileDependency(
    profile: DialectInstance,
    override val descriptor: ProjectDescriptor,
    override val errors: ProjectErrors
) extends ParsedDependency(profile, descriptor, errors) {
  override val scope: DependencyScope = ValidationScope

  val profileName: String = Option(profile.encodes)
    .flatMap(
        _.graph
          .scalarByProperty((Namespace("http://a.ml/vocabularies/amf/core#") + "name").iri())
          .headOption
          .map(_.toString)
    )
    .getOrElse("Unknown")
}

case class ProvidedDependency(project: Gav, provided: Gav, scope: DependencyScope)
