package org.mulesoft.apb.project.client.scala.model

import amf.shapes.client.scala.model.domain.jsonldinstance.{JsonLDObject, JsonLDScalar}
import org.mulesoft.apb.project.client.scala.model.management.SchemaIris.RichString
import DynamicObject._
import scala.language.implicitConversions
import scala.reflect.ClassTag

trait DynamicObject {

  private[apb] val internal: JsonLDObject
  // Getters -----------------------------------------------------------------------------------------------------------
  private def getObjectKernel(property: String)(fallback: => DynamicObject): DynamicObject = {
    internal.graph.getObjectByProperty(property) match {
      case (head: JsonLDObject) :: Nil => toDynamicObject(head)
      case (_: JsonLDScalar) :: Nil    => wrongTypeException(property, "object", "scalar")
      case _ :: _                      => wrongTypeException(property, "object", "array")
      case Nil                         => fallback
    }
  }
  private[apb] def getScalar[T <: Any](property: String)(implicit ct: ClassTag[T]): T = {
    internal.graph.scalarByProperty(property) match {
      case (head: T) :: Nil => head
      case other :: Nil     => wrongTypeException(property, ct.runtimeClass.getSimpleName, other.getClass.getSimpleName)
      case Nil              => noValueForPropertyException(property)
      case array            => wrongTypeException(property, ct.runtimeClass.getSimpleName, array.getClass.getSimpleName)
    }
  }

  private[apb] def getScalarIfPresent[T <: Any](property: String)(implicit ct: ClassTag[T]): Option[T] =
    internal.graph.scalarByProperty(property).collectFirst({ case s: T => s })
  private[apb] def getObject(property: String): DynamicObject = getObjectKernel(property) {
    noValueForPropertyException(property)
  }

  private[apb] def getObjectIfPresent(property: String): Option[DynamicObject] =
    internal.graph.getObjectByProperty(property).collectFirst({ case obj: JsonLDObject => toDynamicObject(obj) })
  private[apb] def getOrCreateObject(property: String): DynamicObject = {
    getObjectKernel(property) {
      val model    = property.toModel
      val internal = toDynamicObject(JsonLDObject.empty(model, model.path))
      withProperty(property, internal)
      internal
    }
  }
  private[apb] def getObjectArray(property: String): Seq[DynamicObject] = {
    internal.graph.getObjectByProperty(property).asInstanceOf[Seq[JsonLDObject]].map(obj => toDynamicObject(obj))
  }

  // Setters
  private[apb] def withProperty(property: String, value: String): this.type = {
    internal.withProperty(property, value)
    this
  }
  private[apb] def withProperty(property: String, value: Int): this.type = {
    internal.withProperty(property, value)
    this
  }
  private[apb] def withProperty(property: String, value: Boolean): this.type = {
    internal.withProperty(property, value)
    this
  }
  private[apb] def withProperty(property: String, value: Float): this.type = {
    internal.withProperty(property, value)
    this
  }

  private[apb] def withProperty(property: String, value: Double): this.type = {
    internal.withProperty(property, value)
    this
  }
  private[apb] def withProperty(property: String, value: DynamicObject): this.type = {
    internal.withProperty(property, value.internal)
    this
  }
  private[apb] def withProperty(property: String, values: Seq[DynamicObject]): this.type = {
    internal.withObjPropertyCollection(property, values.map(_.internal))
    this
  }

  // updaters
  private[apb] def update(property: String)(fn: DynamicObject => Unit): this.type = {
    val start = getOrCreateObject(property)
    fn(start)
    this
  }

  // Exceptions
  private def wrongTypeException(property: String, expected: String, actual: String) =
    throw new IllegalArgumentException(s"Property '$property' is not a '$expected', got '$actual'")
  private def noValueForPropertyException(property: String) =
    throw new IllegalArgumentException(s"Property '$property' has no value")

}

object DynamicObject {
  def apply(model: String): DynamicObject = {
    val model_ = model.toModel
    toDynamicObject(JsonLDObject.empty(model_, model_.path))
  }
  implicit def toDynamicObject(obj: JsonLDObject): DynamicObject = new DynamicObject {
    override private[apb] val internal = obj
  }
  implicit def toJsonLDObject(obj: DynamicObject): JsonLDObject = obj.internal
}
