package org.mulesoft.apb.project.internal.listener
import amf.core.client.scala.errorhandling.AMFErrorHandler
import amf.core.client.scala.parse.document.Reference
import org.mulesoft.apb.project.client.scala.dependency.{ExchangeModulePathHandler, ParsedDependency}
import org.mulesoft.apb.project.client.scala.model.{Gav, ProjectDependency}
import org.mulesoft.apb.project.internal.validations.ProjectValidations.InternalDependencyFileReference
import org.mulesoft.common.client.lexical.SourceLocation

import scala.language.postfixOps

class InternalDependencyFileReferenceListener(mains: Map[Gav, String]) extends RawReferenceListener {

  override protected def checkReferences(references: Seq[Reference], location: String, eh: AMFErrorHandler): Unit = {
    filterDependencyReferences(references)
      .filter(isIncludingInternalFile)
      .foreach(warning(_, eh, location))
  }

  private def pathIsReferenceToInternal(pathHandler: ExchangeModulePathHandler) =
    pathHandler.gav
      .flatMap(Gav.unapply)
      .flatMap(mains.get)
      .exists(mainFile => !refIsEqualToMainFile(pathHandler, mainFile))

  private def refIsEqualToMainFile(refHandler: ExchangeModulePathHandler, mainFile: String) =
    refHandler.relativePath.map(_.stripPrefix("file://")).contains(mainFile)

  private def isIncludingInternalFile(dr: Reference): Boolean =
    buildExchangeHandler(dr).exists(pathIsReferenceToInternal)

  private def buildExchangeHandler(dr: Reference): Option[ExchangeModulePathHandler] =
    ExchangeModulePathHandler.apply(dr.url)

  private def filterDependencyReferences(references: Seq[Reference]) = references.filter(isDependencyReference)

  private def isDependencyReference(reference: Reference) =
    reference.url.contains(ExchangeModulePathHandler.EXCHANGE_MODULES)

  private def warning(reference: Reference, eh: AMFErrorHandler, location: String): Unit = reference.refs.foreach { r =>
    val range = r.reduceToLocation()
    eh.warning(
        InternalDependencyFileReference,
        reference.url,
        s"Detected non root dependency file reference for '${reference.url}'. Only dependency main file should be used from dependant project",
        SourceLocation(location, range.start.line, range.start.column, range.end.line, range.end.column)
    )
  }
}

object InternalDependencyFileReferenceListener {
  def apply(dependencies: Seq[ParsedDependency]): InternalDependencyFileReferenceListener = {
    val mains: Map[Gav, String] = dependencies.map { pd => pd.descriptor.gav -> pd.descriptor.main } toMap

    new InternalDependencyFileReferenceListener(mains)
  }
}
