package org.mulesoft.apb.project.internal.instances

import amf.apicontract.client.scala.AMFConfiguration
import amf.core.client.common.remote.Content
import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.adoption.{IdAdopter, IdAdopterProvider, IdMaker}
import amf.core.client.scala.parse.document.{ParsedDocument, ParserContext, SyamlParsedDocument}
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.plugins.syntax.SyamlSyntaxParsePlugin
import amf.core.internal.remote.FileNotFound
import amf.shapes.client.scala.config.JsonLDSchemaConfiguration
import amf.shapes.client.scala.model.document.{JsonLDInstanceDocument, JsonSchemaDocument}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model.management.Extension
import org.mulesoft.apb.project.client.scala.model.{Gav, ManagementScope, ProjectDependency, ProjectDescriptor}
import org.mulesoft.apb.project.internal.generated.DescriptorSchemaLoader

import java.io.FileNotFoundException
import org.mulesoft.common.collections.Group
import org.yaml.model.YDocument
import org.yaml.parser.YamlParser

import scala.collection.mutable
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future
import scala.reflect.ClassTag

object ExtensionExtractor {
  def apply(gcl: JsonLDInstanceDocument) = gcl.encodes.collect { case obj: JsonLDObject =>
    Extension(obj)
  }
}
case class ScopedExtension(gav: Gav, extension: Extension, isPolicyDefinition: Boolean, dependencies: Set[Gav]) {

  def name: Option[String] = extension.nameIfPresent
}

object ScopedExtension {
  def apply(
      descriptor: ProjectDescriptor,
      gcls: List[JsonLDInstanceDocument],
      isPolicyDefinition: Boolean
  ): Seq[ScopedExtension] = {
    val dependencies = descriptor.dependencies.map(_.gav).toList
    gcls.flatMap { gcl =>
      ExtensionExtractor(gcl).map(extension =>
        ScopedExtension(descriptor.gav, extension, isPolicyDefinition, dependencies.toSet)
      )
    }
  }
}

object ScopedExtensionIndex {
  def apply(extensions: List[ScopedExtension]): ScopedExtensionIndex = {
    val innerMap = extensions.legacyGroupBy(_.gav).mapValues(_.toList)
    ScopedExtensionIndex(innerMap)
  }

  val Empty: ScopedExtensionIndex = ScopedExtensionIndex(List.empty)
}

case class ScopedExtensionIndex(byGav: Map[Gav, List[ScopedExtension]], initial: Set[Gav] = Set.empty) {

  def scoped(allowed: Set[Gav])   = copy(byGav, allowed)
  def addScope(allowed: Set[Gav]) = copy(byGav, initial ++ allowed)

  def findDefinitionFor(name: String): Option[ScopedExtension] = findDefinitionFor(name, initial)
  private def findDefinitionFor(name: String, in: Set[Gav]): Option[ScopedExtension] = {
    findByName(name, in) match {
      case Some(extension: ScopedExtension) if extension.isPolicyDefinition => Some(extension)
      case Some(extension: ScopedExtension) => lookForDefinitionInExtendsChain(extension)
      case None                             => None
    }
  }

  private def lookForDefinitionInExtendsChain(extension: ScopedExtension) = {
    extension.extension.`extends`.view
      .map { extensionName =>
        findDefinitionFor(extensionName, extension.dependencies)
      }
      .collectFirst { case Some(extension) =>
        extension
      }
  }

  def findByName(name: String, in: Set[Gav]) = {
    in.view
      .map { gav: Gav =>
        val found = findByGav(gav)
        val hasName = found.find { extension =>
          extension.extension.nameIfPresent.contains(name)
        }
        hasName
      }
      .collectFirst { case Some(extension) =>
        extension
      }
  }

  def findByGav(gav: Gav) = byGav.getOrElse(gav, Nil)
}

case class ExtensionAssetParser(fetcher: DependencyFetcher, schema: JsonSchemaDocument) {

  private val GCL_FILE_NAME           = "manifest.yaml"
  private val DESCRIPTOR_FILE_NAME    = "exchange.json"
  private val POLICY_SCHEMA_FILE_NAME = "policy-definition.json"

  def parse(descriptor: ProjectDescriptor): Future[ScopedExtensionIndex] = {
    parseDependencies(descriptor, mutable.Set.empty[Gav]).map(ScopedExtensionIndex(_))
  }

  private def parseDependencies(
      descriptor: ProjectDescriptor,
      seen: mutable.Set[Gav]
  ): Future[List[ScopedExtension]] = {
    val toParse = findManagementDependencies(descriptor) ::: findInstanceDependencies(descriptor)
    val all     = toParse.map(dep => parseDependency(dep, seen))
    Future.sequence(all).map(_.flatten)
  }

  private def findInstanceDependencies(descriptor: ProjectDescriptor): List[ProjectDependency] =
    descriptor.instances.flatMap(_.dependencies).toList

  private def findManagementDependencies(descriptor: ProjectDescriptor): List[ProjectDependency] =
    descriptor.dependencies.filter(_.scope == ManagementScope).toList

  private def parseDependency(dependency: ProjectDependency, seen: mutable.Set[Gav]): Future[List[ScopedExtension]] = {
    ifNotSeen(dependency.gav, seen) { () =>
      val loader       = getResourceLoaderForAsset(dependency)
      val descriptor   = parseDescriptor(loader)
      val dependencies = parseDependencies(descriptor, seen)
      val futureGcls   = parseExtension(descriptor, loader)
      chainFutures(dependencies, futureGcls)
    }
  }

  private def parseExtension(futureDescriptor: Future[ProjectDescriptor], loader: ResourceLoader) = {
    for {
      descriptor         <- futureDescriptor
      isPolicyDefinition <- fetchPolicyDefinition(loader)
      extensions         <- parseGcl(loader, descriptor)
    } yield {
      ScopedExtension(descriptor, extensions, isPolicyDefinition).toList
    }
  }

  private def parseGcl(loader: ResourceLoader, descriptor: ProjectDescriptor): Future[List[JsonLDInstanceDocument]] = {
    loader.fetch(GCL_FILE_NAME).map(_.toString()).flatMap { contents =>
      val documents = YamlParser(contents).documents().toList
      val docIndex  = buildGclUriIndex(documents)
      val client    = buildConfig(docIndex).withIdAdopterProvider(getIdAdopterProviderFor(descriptor)).baseUnitClient()

      val parses = docIndex.map { case (uri, _) =>
        client.parseJsonLDInstance(uri, schema).map(_.instance)
      }.toList
      Future.sequence(parses)
    }
  }

  private def getIdAdopterProviderFor(descriptor: ProjectDescriptor) = new IdAdopterProvider {
    override def idAdopter(initialId: String): IdAdopter = new IdAdopter(initialId + "/" + descriptor.gav.path)

    override def idAdopter(initialId: String, idMaker: IdMaker): IdAdopter = new IdAdopter(
        initialId + descriptor.gav.path
    )
  }
  object NestedDocumentRL extends ResourceLoader {

    /** Fetch specified resource and return associated content. Resource should have been previously accepted. If the
      * resource doesn't exists, it returns a failed future caused by a ResourceNotFound exception.
      */
    override def fetch(resource: String): Future[Content] = Future.successful(new Content(resource, resource))

    /** Checks if the resource loader accepts the specified resource. */
    override def accepts(resource: String): Boolean = true
  }

  private def buildGclUriIndex(docs: List[YDocument]) = {
    docs.zipWithIndex.map { case (doc, i) => s"file://gcl-instance/$i.yaml" -> doc }.toMap
  }

  private def buildConfig(docs: Map[String, YDocument]) = {
    val config       = JsonLDSchemaConfiguration.JsonLDSchema()
    val syntaxPlugin = GCLHackedSyntaxPlugin(docs)
    config
      .withPlugin(syntaxPlugin)
      .withResourceLoader(NestedDocumentRL)
  }

  case class GCLHackedSyntaxPlugin(index: Map[String, YDocument]) extends SyamlSyntaxParsePlugin {

    override def parse(text: CharSequence, mediaType: String, ctx: ParserContext): ParsedDocument = SyamlParsedDocument(
        index(text.toString)
    )
  }

  // TODO: we should parse a JsonSchemaDocument here
  private def fetchPolicyDefinition(loader: ResourceLoader) = loader
    .fetch(POLICY_SCHEMA_FILE_NAME)
    .map(_ => true)
    .recover { case _: FileNotFound =>
      false
    }

  private def parseDependencies(
      futureDescriptor: Future[ProjectDescriptor],
      seen: mutable.Set[Gav]
  ): Future[List[ScopedExtension]] = {
    futureDescriptor.flatMap(descriptor => parseDependencies(descriptor, seen))
  }

  private def parseDescriptor(loader: ResourceLoader) = {
    val client = JsonLDSchemaConfiguration.JsonLDSchema().withResourceLoader(loader).baseUnitClient()
    client
      .parseJsonLDInstance(DESCRIPTOR_FILE_NAME, DescriptorSchemaLoader.doc)
      .map(result => asDescriptor(result.instance))
  }

  private def chainFutures(dependencies: Future[List[ScopedExtension]], futureGcl: Future[List[ScopedExtension]]) = {
    Future.sequence(List(futureGcl, dependencies)).map(_.flatten)
  }

  private def getResourceLoaderForAsset(dependency: ProjectDependency) = {
    fetcher.fetch(dependency.groupId, dependency.assetId, dependency.version)
  }

  private def ifNotSeen(gav: Gav, seen: mutable.Set[Gav])(block: () => Future[List[ScopedExtension]]) = {
    if (!seen.contains(gav)) {
      block()
    } else Future.successful(Nil)
  }

  private def asDescriptor(jsonld: JsonLDInstanceDocument) =
    ProjectDescriptor(jsonld.encodes.head.asInstanceOf[JsonLDObject], "")

}
