package org.mulesoft.apb.project.client.scala

import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.validation.AMFValidationResult
import amf.core.internal.unsafe.PlatformSecrets
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model._
import org.mulesoft.apb.project.client.scala.model.descriptor.ProjectDescriptor
import org.mulesoft.apb.project.internal.engine.{AnypointTreeBuilder, LegacyTreeBuilder}
import org.mulesoft.apb.project.internal.generated.DescriptorSchemaLoader.LegacyVersion
import org.mulesoft.apb.project.internal.parser.APBEnv

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

object APIProjectBuilder {

  def apply(dependencyFetcher: DependencyFetcher, loaders: List[ResourceLoader]): APIProjectBuilder = {
    new APIProjectBuilder(dependencyFetcher, loaders)
  }
}

class APIProjectBuilder(
    dependencyFetcher: DependencyFetcher,
    loaders: List[ResourceLoader],
    cacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder
) extends PlatformSecrets {

  def build(project: ProjectDescriptor, errors: List[AMFValidationResult] = Nil): Future[ProjectConfiguration] = {
    val config = project.descriptorVersion match {
      case LegacyVersion => LegacyTreeBuilder(dependencyFetcher, loaders).build(project)
      case _             => AnypointTreeBuilder(dependencyFetcher, loaders, cacheBuilder).build(project)
    }
    config.map(c => c.copy(errors = c.errors.addProjectErrors(errors)))
  }

  def build(project: DescriptorParseResult): Future[ProjectConfiguration] = {
    val DescriptorParseResult(descriptor, report) = project
    build(descriptor, adaptErrorsLocation(report.results.toList))
  }

  def buildFromContent(descriptorAsString: String, directory: Option[String]): Future[ProjectConfiguration] = {
    val descriptor = APBEnv.getProvider(directory).parse(descriptorAsString)
    build(descriptor)
  }

  def buildFromDirectory(directory: String): Future[ProjectConfiguration] = {
    val eventualContent = fetchDescriptorFile(directory)
    eventualContent.flatMap(c => buildFromContent(c.stream.toString, Some(directory)))
  }

  private def adaptErrorsLocation(errors: List[AMFValidationResult]): List[AMFValidationResult] = {
    errors.map(_.copy(location = Some("exchange.json")))
  }

  private def fetchDescriptorFile(directory: String) = {
    val config = AMFGraphConfiguration.empty().withResourceLoaders(loaders)
    platform.fetchContent(directory + "/" + APBEnv.descriptorFileName, config)
  }
}
