package org.mulesoft.apb.project.client.scala

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.apb.internal.plugins.ApbFallbackPlugin
import amf.apicontract.client.scala._
import amf.apicontract.internal.transformation.UnsupportedTransformationPipeline
import amf.core.client.scala.config.UnitCache
import amf.core.client.scala.resource.ResourceLoader
import amf.core.client.scala.transform.TransformationPipelineBuilder
import amf.core.internal.remote.Spec
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptor
import org.mulesoft.apb.project.client.scala.transformation.{
  APBRaml10EditingPipeline,
  JSONDocumentRefCleaningTransformationStage
}
import org.mulesoft.apb.project.internal.config.ConfigProvider
import org.mulesoft.apb.project.internal.dependency.APBResourceLoader
import org.mulesoft.apb.project.internal.engine.MigrationRisks
import org.mulesoft.apb.project.internal.listener.{
  AbsolutePathDetectionRawReferenceListener,
  InternalDependencyFileReferenceListener,
  RawReferenceListener
}

case class ProjectConfiguration(
    designDependencies: Seq[DesignDependency],
    profileDependencies: Seq[ProfileDependency],
    extensionDependencies: Seq[ExtensionDependency],
    dependenciesLoaders: Map[String, ResourceLoader],
    descriptor: ProjectDescriptor,
    validationDialect: Dialect,
    unitCacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder,
    envLoaders: List[ResourceLoader],
    errors: ProjectErrors,
    private[apb] val migrationRisks: MigrationRisks = MigrationRisks.empty
) {

  implicit class AMFConfigPopulator(base: AMFConfiguration) {
    def addExtensions(): AMFConfiguration = extensionDependencies.foldLeft(base) { (acc, curr) =>
      acc.withDialect(curr.dialect)
    }

    def addListeners(): AMFConfiguration = getListeners.foldLeft(base) { (acc, curr) =>
      acc.withEventListener(curr)
    }
  }

  val mainLoader: ResourceLoader = new APBResourceLoader(dependenciesLoaders)
  private lazy val companionLibDependencies: Seq[DesignDependency] =
    extensionDependencies.flatMap(_.companionLibDependency)
  lazy val unitsCache: UnitCache =
    unitCacheBuilder.build(designDependencies ++ companionLibDependencies)

  val classifierConfig: AMFConfiguration = descriptor.classifier
    .map(ConfigProvider.fromClassifier)
    .getOrElse(APIConfiguration.APIWithJsonSchema())
    .withFallback(ApbFallbackPlugin())

  lazy val absolutePathListener: Option[AbsolutePathDetectionRawReferenceListener] =
    AbsolutePathDetectionRawReferenceListener(descriptor)

  val getLegacyConfigListeners: Iterable[RawReferenceListener] = Nil

  def getAnypointConfigListeners: Iterable[RawReferenceListener] =
    absolutePathListener ++ Seq(InternalDependencyFileReferenceListener(designDependencies))

  def getListeners: Iterable[RawReferenceListener] =
    if (!descriptor.isLegacyDescriptor) getAnypointConfigListeners else getLegacyConfigListeners

  def mainFile: String = descriptor.main

  def allDependencies: Seq[ParsedDependency] = designDependencies ++ extensionDependencies ++ profileDependencies

  def getDialectFor(instance: DialectInstance): Option[Dialect] =
    projectParseConfig.configurationState().findDialectFor(instance)

  def apiParseConfig: AMFConfiguration = dependenciesConfig(APIConfiguration.APIWithJsonSchema())

  def projectParseConfig: AMFConfiguration = dependenciesConfig(classifierConfig)

  def configFor(spec: Spec): AMFConfiguration = dependenciesConfig(ConfigProvider.fromSpec(spec))

  def dependenciesConfig(base: AMFConfiguration): AMFConfiguration = addJSONDocumentRefCleaningStage(
      addCompanionLibCheckStage(
          base
            .withResourceLoaders(List(mainLoader) ++ envLoaders)
            .withDialect(validationDialect)
            .withUnitCache(unitsCache)
            .addListeners()
            .addExtensions()
      )
  )

  private def addCompanionLibCheckStage(base: AMFConfiguration): AMFConfiguration = {
    if (descriptor.isRaml)
      base.withTransformationPipeline(new APBRaml10EditingPipeline(companionLibDependencies))
    else base
  }

  private def addJSONDocumentRefCleaningStage(base: AMFConfiguration): AMFConfiguration = {
    val cleaningTransformationStage = new JSONDocumentRefCleaningTransformationStage()
    val newPipelines = base
      .configurationState()
      .getTransformationPipelines()
      .map(pipeline =>
        if (pipeline.isInstanceOf[UnsupportedTransformationPipeline])
          pipeline
        else
          TransformationPipelineBuilder.fromPipeline(pipeline).prepend(cleaningTransformationStage).build()
      )

    val configuration = newPipelines.foldLeft(base) { (config, pipeline) =>
      config.withTransformationPipeline(pipeline)
    }

    configuration
  }
}
