package org.mulesoft.apb.project.client.scala.model.descriptor

import amf.core.client.scala.model.{BoolField, StrField}
import amf.core.internal.parser.domain.Fields
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.apb.project.client.scala.model.ProjectDependencyModel.{AssetId, GroupId, Version, Classifier}
import org.mulesoft.apb.project.client.scala.model.ProjectDescriptorModel.{Metadata => MetadataIri, _}
import org.mulesoft.apb.project.client.scala.model.descriptor.documentation.Documentation
import org.mulesoft.apb.project.client.scala.model._
import org.mulesoft.apb.project.internal.generated.DescriptorSchemaLoader.LegacyVersion
import org.mulesoft.common.time.SimpleDateTime

import scala.language.implicitConversions

case class ProjectDescriptor private[apb] (override private[apb] val wrapped: JsonLDObject)
    extends JsonLDObjectWrapper
    with ProjectBase {

  override type This = ProjectDescriptor
  private implicit def metadataConversion(obj: JsonLDObject): Option[Metadata]    = Option(obj).map(Metadata.apply)
  private implicit def instanceConversion(objs: Seq[JsonLDObject]): Seq[Instance] = objs.map(Instance.apply)
  private implicit def documentationConversion(objs: Seq[JsonLDObject]): Seq[Documentation] =
    objs.map(Documentation.apply)

  def main: Option[String]                  = get[StrField](ProjectDescriptorModel.Main)
  def descriptorVersion: String             = getDescriptorVersion
  def instances: Seq[Instance]              = get[Seq[JsonLDObject]](ApiInstances)
  def originalFormatVersion: Option[String] = get[StrField](OriginalFormatVersion)
  def metadata: Option[Metadata]            = get[JsonLDObject](MetadataIri)
  def backwardsCompatible: Option[Boolean]  = get[BoolField](BackwardsCompatible)
  def publishWithRefFiles: Option[Boolean]  = get[BoolField](PublishWithRefFiles)
  def description: Option[String]           = get[StrField](Description)
  def projectId: Option[String]             = get[StrField](ProjectId)
  def documentation: Seq[Documentation]     = get[Seq[JsonLDObject]](ProjectDescriptorModel.Documentation)
  override protected def fields: Fields     = wrapped.fields

  private def getDescriptorVersion: String = {
    val version: StrField = get(DescriptorVersion)
    version.option() match {
      case Some("1.0.0")       => "1.0.0"
      case Some(LegacyVersion) => LegacyVersion
      case Some(_)             => "1.0.0"
      case None                => LegacyVersion
    }
  }

  private[apb] def isRaml: Boolean = classifier.contains("raml")

  def isLegacyDescriptor: Boolean = descriptorVersion == LegacyVersion

  def withMain(main: String): ProjectDescriptor = {
    val next = wrapped.withProperty(ProjectDescriptorModel.Main, main)
    this
  }
  def withDependency(dependency: ProjectDependency): ProjectDescriptor = {
    val newDependencies = dependencies.filterNot(_.gav == dependency.gav) :+ dependency
    wrapped.withObjPropertyCollection(Dependencies, newDependencies.map(_.wrapped))
    this
  }

  def withInstance(instance: Instance): ProjectDescriptor = {
    val newInstances = instances.filterNot(_.gcl == instance.gcl) :+ instance
    withInstances(newInstances)
  }

  def withInstances(instances: Seq[Instance]): ProjectDescriptor = {
    wrapped.withObjPropertyCollection(ApiInstances, instances.map(_.wrapped))
    this
  }

  def withName(name: String): ProjectDescriptor = {
    wrapped.withProperty(Name, name)
    this
  }
  def withDescriptorVersion(version: String): ProjectDescriptor = {
    wrapped.withProperty(DescriptorVersion, version)
    this
  }
  def withGav(gav: Gav): ProjectDescriptor = {
    wrapped
      .withProperty(GroupId, gav.groupId)
      .withProperty(AssetId, gav.assetId)
    withVersion(gav.version)
    this
  }

  def withVersion(version: String): ProjectDescriptor = {
    wrapped.withProperty(Version, version)
    this
  }
  def withClassifier(classifier: String): ProjectDescriptor = {
    wrapped.withProperty(Classifier, classifier)
    this
  }
  def withTags(tags: Seq[String]): ProjectDescriptor = {
    wrapped.withStringPropertyCollection(Tags, tags)
    this
  }
  def withTag(tag: String): ProjectDescriptor = withTags(tags :+ tag)
  def withApiVersion(apiVersion: String): ProjectDescriptor = {
    wrapped.withProperty(ApiVersion, apiVersion)
    this
  }
  def withOrganizationId(organizationId: String): ProjectDescriptor = {
    wrapped.withProperty(OrganizationId, organizationId)
    this
  }
  def withOriginalFormatVersion(originalFormatVersion: String): ProjectDescriptor = {
    wrapped.withProperty(OriginalFormatVersion, originalFormatVersion)
    this
  }

  def withMetadata(metadata: Metadata): ProjectDescriptor = {
    wrapped.withProperty(MetadataIri, metadata.wrapped)
    this
  }
  def withBackwardsCompatible(value: Boolean): ProjectDescriptor = {
    wrapped.withProperty(BackwardsCompatible, value)
    this
  }
  def withPublishWithRefFiles(value: Boolean): ProjectDescriptor = {
    wrapped.withProperty(PublishWithRefFiles, value)
    this
  }
  def withDescription(description: String): ProjectDescriptor = {
    wrapped.withProperty(Description, description)
    this
  }
  def withProjectId(projectId: String): ProjectDescriptor = {
    wrapped.withProperty(ProjectId, projectId)
    this
  }

  def withStatus(status: String): ProjectDescriptor = {
    wrapped.withProperty(Status, status)
    this
  }
  def withCreatedAt(createdAt: SimpleDateTime): ProjectDescriptor = {
    wrapped.withDateTimeProperty(CreatedAt, createdAt)
    this
  }
  def withCreatedBy(createdBy: String): ProjectDescriptor = {
    wrapped.withProperty(CreatedBy, createdBy)
    this
  }

  def withUpdatedAt(updatedAt: SimpleDateTime): ProjectDescriptor = {
    wrapped.withDateTimeProperty(UpdatedAt, updatedAt)
    this
  }

  def withTenantId(tenantId: String): ProjectDescriptor = {
    wrapped.withProperty(TenantId, tenantId)
    this
  }

  def withCategories(categories: Seq[String]): ProjectDescriptor = {
    wrapped.withStringPropertyCollection(Categories, categories)
    this
  }

  def withFields(fields: Seq[String]): ProjectDescriptor = {
    wrapped.withStringPropertyCollection(CustomFields, fields)
    this
  }

  def withKeyValueTags(keyValueTags: Seq[String]): ProjectDescriptor = {
    wrapped.withStringPropertyCollection(KeyValueTags, keyValueTags)
    this
  }

  def withType(`type`: String): ProjectDescriptor = {
    wrapped.withProperty(TypeField, `type`)
    this
  }

  def withMajorVersionComponent(version: Int): ProjectDescriptor = {
    wrapped.withProperty(MajorVersionComponent, version)
    this
  }
  def withMinorVersionComponent(version: Int): ProjectDescriptor = {
    wrapped.withProperty(MinorVersionComponent, version)
    this
  }
  def withPatchVersionComponent(version: Int): ProjectDescriptor = {
    wrapped.withProperty(PatchVersionComponent, version)
    this
  }
  def withApiId(apiId: String): ProjectDescriptor = {
    wrapped.withProperty(ApiId, apiId)
    this
  }
  def withProjectType(projectType: String): ProjectDescriptor = {
    wrapped.withProperty(ProjectType, projectType)
    this
  }
  def withDocumentation(doc: Documentation): ProjectDescriptor = {
    val newInstances = documentation.filterNot(_.name.contains(doc.name.getOrElse(""))) :+ doc
    withDocumentation(newInstances)
  }

  def withDocumentation(docs: Seq[Documentation]): ProjectDescriptor = {
    wrapped.withObjPropertyCollection(ProjectDescriptorModel.Documentation, docs.map(_.wrapped))
    this
  }

  def withContactName(contactName: String): ProjectDescriptor = {
    wrapped.withProperty(ContactName, contactName)
    this
  }

  def withContactEMail(contactEMail: String): ProjectDescriptor = {
    wrapped.withProperty(ContactEMail, contactEMail)
    this
  }
}

object ProjectDescriptor {
  def apply(main: String): ProjectDescriptor = {
    val inner = createInner().set(Main, main)
    new ProjectDescriptor(inner)
  }

  def apply(): ProjectDescriptor = new ProjectDescriptor(createInner())

  private def createInner() = {
    JsonLDObject.empty(ProjectDescriptorModel.entityModel, JsonPath.empty)
  }
}
