package org.mulesoft.apb.project.client.scala.config

import amf.apb.internal.plugins.ApbFallbackPlugin
import amf.apicontract.client.scala.{AMFConfiguration, APIConfiguration}
import amf.apicontract.internal.transformation.UnsupportedTransformationPipeline
import amf.core.internal.remote.Spec
import org.mulesoft.apb.project.client.scala.ProjectConfiguration
import org.mulesoft.apb.project.client.scala.dependency.{APBUnitCacheBuilder, UnitCacheBuilder}
import org.mulesoft.apb.project.client.scala.model.descriptor.ProjectDescriptor
import org.mulesoft.apb.project.internal.dependency.config.ConfigProvider
import org.mulesoft.apb.project.internal.idadoption.IdAdopterProvider
import org.mulesoft.apb.project.internal.transformation.APBTransformationPipeline

object AMFConfigurationFactory {

  def createFromClassifier(descriptor: ProjectDescriptor): AMFConfiguration = {
    descriptor
      .classifier()
      .map(ConfigProvider.fromClassifier)
      .getOrElse(APIConfiguration.APIWithJsonSchema())
      .withFallback(ApbFallbackPlugin())
  }

  def createFrom(
      config: ProjectConfiguration,
      base: AMFConfiguration,
      cacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder
  ): AMFConfiguration =
    dependenciesConfig(config, base, cacheBuilder)

  def createSpecAgnosticParsingConfig(
      config: ProjectConfiguration,
      cacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder
  ): AMFConfiguration = {
    dependenciesConfig(config, APIConfiguration.APIWithJsonSchema(), cacheBuilder)
  }

  def createSpecificationParsingConfig(
      config: ProjectConfiguration,
      cacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder
  ): AMFConfiguration = {
    dependenciesConfig(config, createFromClassifier(config.descriptor), cacheBuilder)
  }

  private[apb] def createFromSpec(
      config: ProjectConfiguration,
      spec: Spec,
      cacheBuilder: UnitCacheBuilder = APBUnitCacheBuilder
  ): AMFConfiguration = {
    dependenciesConfig(config, ConfigProvider.fromSpec(spec), cacheBuilder)
  }

  private def dependenciesConfig(
      config: ProjectConfiguration,
      base: AMFConfiguration,
      cacheBuilder: UnitCacheBuilder
  ): AMFConfiguration = {
    base
      .withResourceLoaders(List(config.mainLoader()) ++ config.envLoaders)
      .withDialect(config.validationDialect)
      .withUnitCache(buildUnitCache(config, cacheBuilder))
      .addListeners(config)
      .addExtensions(config)
      .withIdAdopterProvider(IdAdopterProvider)
      .withPipelines(config)
  }

  private def buildUnitCache(config: ProjectConfiguration, cacheBuilder: UnitCacheBuilder) = {
    cacheBuilder.build(config.designDependencies ++ config.extensionDependencies.flatMap(_.companionLibDependency))
  }

  implicit class AMFConfigPopulator(base: AMFConfiguration) {
    def addExtensions(config: ProjectConfiguration): AMFConfiguration = config.extensionDependencies.foldLeft(base) {
      (acc, curr) =>
        acc.withDialect(curr.dialect)
    }

    def addListeners(config: ProjectConfiguration): AMFConfiguration =
      config.getListeners().foldLeft(base) { (acc, curr) =>
        acc.withEventListener(curr)
      }

    def withPipelines(config: ProjectConfiguration): AMFConfiguration = overwritePipelines(config, base)
  }

  private def overwritePipelines(config: ProjectConfiguration, base: AMFConfiguration) = {
    val newPipelines = base.configurationState().getTransformationPipelines().flatMap {
      case unsupported: UnsupportedTransformationPipeline => Option(unsupported)
      case pipeline                                       => Option(APBTransformationPipeline(config, pipeline))
    }
    newPipelines.foldLeft(base) { (config, pipeline) => config.withTransformationPipeline(pipeline) }
  }
}
