package org.mulesoft.apb.project.internal.model.project

import amf.core.client.scala.model.document.BaseUnit
import amf.core.client.scala.validation.AMFValidationResult
import amf.core.internal.metamodel.Field
import amf.core.internal.metamodel.Type.Str
import amf.core.internal.metamodel.document.DocumentModel.Encodes
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.project.client.scala.{DependencySet, ProjectConfiguration, ProjectErrors}
import org.mulesoft.apb.project.client.scala.extensions.APIProjectExtension
import org.mulesoft.apb.project.client.scala.model.ProjectBuildResult
import org.mulesoft.apb.project.client.scala.model.descriptor.ProjectDescriptor
import org.mulesoft.apb.project.client.scala.model.project.{Project, ProjectInfo}
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.Management
import org.mulesoft.apb.project.internal.idadoption.APBIdAdopter
import org.mulesoft.apb.project.internal.metamodel.ProjectInfoModel.{Contract, Instances}
import org.mulesoft.apb.project.internal.model.ProjectDescriptorModel.Documentation
import org.mulesoft.apb.project.internal.view.ProjectView

object ProjectDocumentBuilder {
  val INSTANCE_ID_IRI: Field = Field(Str, Management + "instanceID")

  private[apb] def apply(configuration: DependencySet): ProjectDocumentBuilder = {
    new ProjectDocumentBuilder(configuration.descriptor()).withProjectErrors(configuration.errors())
  }
}

// TODO: internal
case class ProjectDocumentBuilder(descriptor: ProjectDescriptor) {
  private val projectInfo                          = ProjectInfo()
  private val projectDocument                      = ProjectDocument().set(Encodes, projectInfo)
  private var extensions: Seq[APIProjectExtension] = List.empty
  private var contract: Option[BaseUnit]           = None
  private var errors: Seq[AMFValidationResult]     = Seq.empty
  private var instances: Seq[JsonLDObject]         = Seq.empty
  private var documentation: Seq[JsonLDObject]     = Seq.empty
  private var base: Option[String]                 = None

  def withContract(unit: BaseUnit, errors: Seq[AMFValidationResult]): ProjectDocumentBuilder = {
    contract = Some(unit)
    this.errors = this.errors ++ errors
    this
  }

  def withInstances(instances: List[JsonLDObject], errors: Seq[AMFValidationResult]): ProjectDocumentBuilder = {
    this.instances = this.instances ++ instances
    this.errors = this.errors ++ errors
    this
  }

  def withDocumentation(documentation: List[JsonLDObject], errors: Seq[AMFValidationResult]): ProjectDocumentBuilder = {
    this.documentation = this.documentation ++ documentation
    this.errors = this.errors ++ errors
    this
  }

  def withExtensions(extensions: Seq[APIProjectExtension]) = {
    this.extensions = extensions
    this
  }

  def withBase(base: String): ProjectDocumentBuilder = {
    this.base = Some(base)
    this
  }

  def withProjectErrors(errors: ProjectErrors): ProjectDocumentBuilder = {
    this.errors = errors.allErrors
    this
  }
  def build(): ProjectBuildResult = {
    addFieldsFromDescriptor(projectInfo, descriptor)
    val project = Project(projectDocument)
    adoptProject()
    // We must honor ids because they might come from cache. This means that ID generation of each part will be isolated to each.
    contract.foreach(projectInfo.setWithoutId(Contract, _))
    if (instances.nonEmpty) projectInfo.setArrayWithoutId(Instances, instances)
    if (documentation.nonEmpty) projectInfo.setArrayWithoutId(Documentation, documentation)
    extensions.foreach(extension => extension.extend(project))
    ProjectBuildResult(project, project.projectInfo.id, this.errors)
  }

  private def adoptProject(): Unit = {
    val finalBase = this.base.getOrElse("http://project.aml")
    new APBIdAdopter(finalBase).adoptFromRoot(projectDocument)
  }

  private def addFieldsFromDescriptor(projectInfo: ProjectInfo, descriptor: ProjectDescriptor) = {
    ProjectView(projectInfo).view(descriptor.internal)
  }
}
