package org.mulesoft.apb.project.client.scala.dependency

import amf.aml.client.scala.model.document.{Dialect, DialectInstance}
import amf.core.client.scala.model.document.{BaseUnit, Module}
import amf.core.client.scala.vocabulary.Namespace
import org.mulesoft.apb.project.client.scala.ProjectErrors
import org.mulesoft.apb.project.client.scala.model._
import org.mulesoft.apb.project.client.scala.model.descriptor.{
  DependencyScope,
  DesignScope,
  ExtensionScope,
  Gav,
  ProjectDescriptor,
  ValidationScope
}
import org.mulesoft.apb.project.internal.engine.MigrationRisks
import org.mulesoft.apb.project.internal.listener.AbsolutePathDetectionRawReferenceListener

trait Dependency {
  val descriptor: ProjectDescriptor
  val errors: ProjectErrors
}
case class IgnoredDependency(descriptor: ProjectDescriptor, errors: ProjectErrors) extends Dependency

abstract class ParsedDependency private[apb] (
    val baseUnit: BaseUnit,
    val descriptor: ProjectDescriptor,
    val errors: ProjectErrors
) extends Dependency {
  def scope(): DependencyScope
  def location(): String = baseUnit.location().getOrElse(baseUnit.id)
}

case class DesignDependency private[apb] (
    override val baseUnit: BaseUnit,
    override val descriptor: ProjectDescriptor,
    override val errors: ProjectErrors = ProjectErrors.EMPTY,
    private[apb] val listener: Option[AbsolutePathDetectionRawReferenceListener] = None,
    private[apb] val risks: MigrationRisks = MigrationRisks.empty
) extends ParsedDependency(baseUnit, descriptor, errors) {
  override val scope: DependencyScope = DesignScope

  private[apb] def update(migrationRisks: MigrationRisks): DesignDependency =
    this.copy(risks = migrationRisks)
}

case class ExtensionDependency private[apb] (
    dialect: Dialect,
    companionLib: Option[(String, Module)],
    override val descriptor: ProjectDescriptor,
    override val errors: ProjectErrors
) extends ParsedDependency(dialect, descriptor, errors) {
  override val scope: DependencyScope = ExtensionScope

  def companionLibDependency(): Option[DesignDependency] =
    companionLib.map { case (name, library) =>
      DesignDependency(library, descriptor.copy().withMain(name), errors)
    }
}

case class ProfileDependency private[apb] (
    profile: DialectInstance,
    override val descriptor: ProjectDescriptor,
    override val errors: ProjectErrors
) extends ParsedDependency(profile, descriptor, errors) {
  override val scope: DependencyScope = ValidationScope

  def profileName(): String = extractProfileName().getOrElse("Unknown")

  private def extractProfileName() = {
    Option(profile.encodes)
      .flatMap(
          _.graph
            .scalarByProperty((Namespace("http://a.ml/vocabularies/amf/core#") + "name").iri())
            .headOption
            .map(_.toString)
      )
  }
}

case class ProvidedDependency(project: Gav, provided: Gav, scope: DependencyScope)

case class InvalidParsedDependency private[apb] (
    override val baseUnit: BaseUnit,
    override val descriptor: ProjectDescriptor,
    override val errors: ProjectErrors
) extends ParsedDependency(baseUnit, descriptor, errors) {
  override val scope: DependencyScope = DesignScope
}
