package org.mulesoft.apb.project.internal.instances

import amf.core.client.scala.model.domain.DomainElement
import amf.core.client.scala.vocabulary.ValueType
import amf.shapes.client.scala.model.document.JsonLDInstanceDocument
import amf.shapes.client.scala.model.domain.jsonldinstance.{JsonLDElement, JsonLDObject}
import APIInstanceModel._
import org.mulesoft.apb.project.client.scala.model.project.management.SchemaIris.{ENVIRONMENT_TARGET_REF, NAME}
import org.mulesoft.apb.project.client.scala.model.project.management._

import scala.collection.mutable

case class ResourceCatalog(instances: List[JsonLDInstanceDocument]) {
  private val resourcesIndex: Map[ResourceKind, List[JsonLDInstanceDocument]] = catalogResources()

  val policiesIndex: PolicyBindingIndex  = PolicyBindingIndex.build(this)
  val environmentIndex: EnvironmentIndex = EnvironmentIndex.build(this)

  def getPolicyBindings: List[PolicyBinding] = {
    getResource(ResourceKind.PolicyBinding)
      .map(PolicyBinding(_))
      .filter(_.hasTarget)
  }

  def getEnvironmentBindings: List[EnvironmentBinding] = {
    getResource(ResourceKind.EnvironmentBinding)
      .map(EnvironmentBinding(_))
      .filter(_.hasTarget)
  }

  def getEnvironments: List[Environment] = {
    getResource(ResourceKind.Environment)
      .map(Environment(_))
  }

  def getServices: List[Service] = {
    getResource(ResourceKind.Service)
      .map(Service(_))
  }

  def getApiInstances: List[APIInstance] = {
    getResource(ResourceKind.ApiInstance)
      .map(APIInstance(_))
  }

  private def getResource(kind: ResourceKind) = {
    resourcesIndex
      .get(kind)
      .toList
      .flatten
      .flatMap(getEncoded)
  }

  private def getEncoded(document: JsonLDInstanceDocument) = {
    document.encodes.collectFirst { case jsonLdObject: JsonLDObject => jsonLdObject }
  }

  private def catalogResources(): Map[ResourceKind, List[JsonLDInstanceDocument]] = instances.groupBy(getResourceKind)

  private def getResourceKind(document: JsonLDInstanceDocument): ResourceKind = {
    if (document.encodes.length == 1) getResourceKind(document.encodes.head)
    else ResourceKind.Other
  }

  private def getResourceKind(element: JsonLDElement) = element match {
    case e: DomainElement if ResourceCatalog.isApiInstanceNode(e) => ResourceKind.ApiInstance
    case e: DomainElement if isPolicyBindingNode(e)               => ResourceKind.PolicyBinding
    case e: DomainElement if isServiceNode(e)                     => ResourceKind.Service
    case e: DomainElement if isEnvironmentNode(e)                 => ResourceKind.Environment
    case e: DomainElement if isEnvironmentBindingNode(e)          => ResourceKind.EnvironmentBinding
    case e: DomainElement if isExtensionNode(e)                   => ResourceKind.Extension
    case _                                                        => ResourceKind.Other
  }

  private def isPolicyBindingNode(encodes: DomainElement) = isTermNode(encodes, POLICY_BINDING_TERM)

  private def isServiceNode(encodes: DomainElement) = isTermNode(encodes, SERVICE_TERM)

  private def isEnvironmentNode(encodes: DomainElement) = isTermNode(encodes, ENVIRONMENT_TERM)

  private def isEnvironmentBindingNode(encodes: DomainElement) = isTermNode(encodes, ENVIRONMENT_BINDING_TERM)

  private def isExtensionNode(encodes: DomainElement) = isTermNode(encodes, EXTENSION_TERM)

  private def isTermNode(element: DomainElement, term: ValueType) = element.meta.typeIris.contains(term.iri())

}

object ResourceCatalog {
  def isApiInstanceNode(encodes: DomainElement)                   = isTermNode(encodes, API_INSTANCE_TERM)
  private def isTermNode(element: DomainElement, term: ValueType) = element.meta.typeIris.contains(term.iri())
}
