package org.mulesoft.apb.project.client.scala.extensions

import amf.aml.internal.metamodel.document.DialectInstanceModel.DefinedBy
import amf.core.client.scala.model.document.Document
import amf.core.internal.metamodel.Field
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.domain.metamodel.jsonldschema.JsonLDEntityModel
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.apb.project.client.scala.extensions.EnvironmentExtension.ENVIRONMENT_IRI
import org.mulesoft.apb.project.client.scala.extensions.environment.InstanceEnvironment
import org.mulesoft.apb.project.client.scala.model.project.Project
import org.mulesoft.apb.project.client.scala.model.project.management.APIInstance
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.Management
import org.mulesoft.apb.project.internal.extensions.environment.InstanceEnvironmentModel
import org.mulesoft.apb.project.internal.extensions.environment.InstanceEnvironmentModel.{Id, Name, Type}
import org.mulesoft.apb.project.internal.metamodel.ProjectInfoModel.Instances
import org.mulesoft.apb.project.internal.model.project.ProjectDocumentBuilder.INSTANCE_ID_IRI

sealed trait APIProjectExtension {

  def extend(project: Project): Unit
}

object EnvironmentExtension {
  private[apb] val ENVIRONMENT_IRI = Field(InstanceEnvironmentModel, Management + "environment")
}

case class EnvironmentExtension(envs: Seq[InstanceEnvironment]) extends APIProjectExtension {

  private val grouped = envs.collect {
    case env if env.environmentId().nonEmpty => env.environmentId().value() -> env
  }.toMap

  override def extend(project: Project): Unit = {
    val next = project.instances().map(setInstanceId).map(setEnvironment)
    project.projectInfo.setArrayWithoutId(Instances, next.map(_.internal))
  }

  private def setEnvironment(instance: APIInstance) = {
    instance
      .getStringLabelIfPresent("mulesoft.com/environment-id")
      .map { envId =>
        val enhanced = grouped.get(envId).map { env =>
          val clonedEnv = asJsonLd(env, instance.id)
          instance.withProperty(ENVIRONMENT_IRI.value.iri(), clonedEnv)
        }
        enhanced.getOrElse(instance)
      }
      .getOrElse(instance)
  }

  private def setInstanceId(instance: APIInstance) = {
    instance
      .getIntLabelIfPresent("mulesoft.com/apiinstance-id")
      .map(id => instance.withProperty(INSTANCE_ID_IRI.value.iri(), id))
      .getOrElse(instance)
  }

  private def asJsonLd(environment: InstanceEnvironment, parentId: String): JsonLDObject = {
    val path = JsonPath.empty.concat(environment.environmentId().option().getOrElse("environment"))
    val result =
      JsonLDObject.empty(JsonLDEntityModel(environment.meta.`type`, List.empty, path), path).adopted(parentId)
    environment
      .`type`()
      .option()
      .foreach(value => result.withProperty(Type.value.iri(), value))
    environment.environmentId().option().foreach(value => result.withProperty(Id.value.iri(), value))
    environment.name().option().foreach(value => result.withProperty(Name.value.iri(), value))
    result
  }
}

case class APIContractSchemaExtension(schemaURI: String) extends APIProjectExtension {
  override def extend(project: Project): Unit = {
    Option(project.apiContract()).foreach({ case d: Document => d.set(DefinedBy, schemaURI) })
  }
}
