package org.mulesoft.apb.project.internal.model

import amf.shapes.client.scala.model.domain.jsonldinstance.{JsonLDObject, JsonLDScalar}
import org.mulesoft.apb.project.client.scala.model.project.management.SchemaIris.RichString
import GraphAccessors._
import org.mulesoft.apb.project.client.scala.model.DynamicObject

import scala.reflect.ClassTag

trait GraphAccessors extends DynamicObject {

  private[apb] val internal: JsonLDObject

  // Getters -----------------------------------------------------------------------------------------------------------
  private def getObjectKernel(property: String)(fallback: => GraphAccessors): GraphAccessors = {
    internal.graph.getObjectByProperty(property) match {
      case (head: JsonLDObject) :: Nil => toDynamicObject(head)
      case (_: JsonLDScalar) :: Nil    => wrongTypeException(property, "object", "scalar")
      case _ :: _                      => wrongTypeException(property, "object", "array")
      case Nil                         => fallback
    }
  }

  private[apb] def getScalar[T <: Any](property: String)(implicit ct: ClassTag[T]): T = {
    internal.graph.scalarByProperty(property) match {
      case (head: T) :: Nil => head
      case other :: Nil     => wrongTypeException(property, ct.runtimeClass.getSimpleName, other.getClass.getSimpleName)
      case Nil              => noValueForPropertyException(property)
      case array            => wrongTypeException(property, ct.runtimeClass.getSimpleName, array.getClass.getSimpleName)
    }
  }

  private[apb] def getScalarIfPresent[T <: Any](property: String)(implicit ct: ClassTag[T]): Option[T] =
    internal.graph.scalarByProperty(property).collectFirst({ case s: T => s })

  private[apb] def getObject(property: String): GraphAccessors = getObjectKernel(property) {
    noValueForPropertyException(property)
  }

  private[apb] def containsProperty(property: String): Boolean = internal.graph.containsProperty(property)

  private[apb] def getObjectIfPresent(property: String): Option[GraphAccessors] =
    internal.graph.getObjectByProperty(property).collectFirst({ case obj: JsonLDObject => toDynamicObject(obj) })

  private[apb] def getOrCreateObject(property: String): GraphAccessors = {
    getObjectKernel(property) {
      val model  = property.toModel
      val id     = internal.id + "/" + model.path.lastSegment.getOrElse(model.path.toString())
      val newObj = toDynamicObject(JsonLDObject.empty(model, model.path).withId(id))
      withProperty(property, newObj)
      newObj
    }
  }

  private[apb] def getScalarArray[T <: Any](property: String)(implicit ct: ClassTag[T]): Seq[T] = {
    internal.graph.scalarByProperty(property).collect { case s: T => s }
  }

  private[apb] def getObjectArray(property: String): Seq[GraphAccessors] = {
    internal.graph.getObjectByProperty(property).asInstanceOf[Seq[JsonLDObject]].map(obj => toDynamicObject(obj))
  }

  // Setters
  private[apb] def withProperty(property: String, value: String): this.type = {
    internal.withProperty(property, value)
    this
  }

  private[apb] def withProperty(property: String, value: Int): this.type = {
    internal.withProperty(property, value)
    this
  }

  private[apb] def withProperty(property: String, value: Boolean): this.type = {
    internal.withProperty(property, value)
    this
  }

  private[apb] def withProperty(property: String, value: Float): this.type = {
    internal.withProperty(property, value)
    this
  }

  private[apb] def withProperty(property: String, value: Double): this.type = {
    internal.withProperty(property, value)
    this
  }

  private[apb] def withProperty(property: String, value: GraphAccessors): this.type = {
    internal.withProperty(property, value.internal)
    this
  }

  private[apb] def withProperty(property: String, values: Seq[GraphAccessors]): this.type = {
    internal.withObjPropertyCollection(property, values.map(_.internal))
    this
  }

  // updaters
  private[apb] def update(property: String)(fn: GraphAccessors => Unit): this.type = {
    val start = getOrCreateObject(property)
    fn(start)
    this
  }

  // Exceptions
  private def wrongTypeException(property: String, expected: String, actual: String) =
    throw new IllegalArgumentException(s"Property '$property' is not a '$expected', got '$actual'")

  private def noValueForPropertyException(property: String) =
    throw new IllegalArgumentException(s"Property '$property' has no value")

}

object GraphAccessors {
  def apply(model: String): GraphAccessors = {
    val model_ = model.toModel
    toDynamicObject(JsonLDObject.empty(model_, model_.path))
  }
  implicit def toDynamicObject(obj: JsonLDObject): GraphAccessors = new GraphAccessors {
    override private[apb] val internal = obj
  }
  implicit def toJsonLDObject(obj: GraphAccessors): JsonLDObject = obj.internal
}
