package org.mulesoft.apb.project.client.scala.extensions

import amf.core.client.scala.model.StrField
import amf.core.client.scala.model.domain.{AmfObject, DomainElement}
import amf.core.client.scala.vocabulary.ValueType
import amf.core.internal.metamodel.Type.Str
import amf.core.internal.metamodel.{Field, ModelDefaultBuilder, Obj}
import amf.core.internal.parser.domain.{Annotations, Fields}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.domain.metamodel.jsonldschema.JsonLDEntityModel
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.apb.project.client.scala.extensions.EnvironmentExtension.ENVIRONMENT_IRI
import org.mulesoft.apb.project.client.scala.extensions.InstanceEnvironmentModel.{Id, Name}
import org.mulesoft.apb.project.client.scala.model.ProjectDocumentBuilder.INSTANCE_ID_IRI
import org.mulesoft.apb.project.client.scala.model.{Project, ProjectInfo}
import org.mulesoft.apb.project.client.scala.model.management.APIInstance
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.{Anypoint, Core, Management}
import org.mulesoft.apb.project.internal.metamodel.ProjectInfoModel.Instances

sealed trait APIProjectExtension {

  def extend(project: Project): Project
}

object EnvironmentExtension {
  val ENVIRONMENT_IRI = Field(InstanceEnvironmentModel, Management + "environment")
}

case class EnvironmentExtension(envs: List[InstanceEnvironment]) extends APIProjectExtension {

  private val grouped = envs.collect {
    case env if env.environmentId().nonEmpty => env.environmentId().value() -> env
  }.toMap

  override def extend(project: Project): Project = {
    val next = project.instances().map(setInstanceId).map(setEnvironment)
    project.projectInfo.setArray(Instances, next)
    project
  }

  private def setEnvironment(instance: JsonLDObject) = {
    val query = APIInstance(instance)
    query.getStringLabelIfPresent("mulesoft.com/environment-id")
      .map { envId =>
        val enhanced = grouped.get(envId).map { env =>
          val clonedEnv = asJsonLd(env)
          instance.withProperty(ENVIRONMENT_IRI.value.iri(), clonedEnv)
        }
        enhanced.getOrElse(instance)
    }.getOrElse(instance)
  }

  private def setInstanceId(instance: JsonLDObject) = {
    APIInstance(instance).getIntLabelIfPresent("mulesoft.com/apiinstance-id")
      .map(id => instance.withProperty(INSTANCE_ID_IRI.value.iri(), id))
      .getOrElse(instance)
  }

  private def asJsonLd(environment: InstanceEnvironment): JsonLDObject = {
    val path   = JsonPath.empty.concat(environment.environmentId().option().getOrElse("environment"))
    val result = JsonLDObject.empty(JsonLDEntityModel(environment.meta.`type`, List.empty, path), path)
    environment.`type`().option().foreach(value => result.withProperty(InstanceEnvironmentModel.Type.value.iri(), value))
    environment.environmentId().option().foreach(value => result.withProperty(Id.value.iri(), value))
    environment.name().option().foreach(value => result.withProperty(Name.value.iri(), value))
    result
  }
}

case class InstanceEnvironment private[apb] (fields: Fields, annotations: Annotations) extends DomainElement {
  override def meta: Obj = InstanceEnvironmentModel

  def environmentId(): StrField = fields.field(Id)
  def name(): StrField          = fields.field(Name)
  def `type`(): StrField        = fields.field(InstanceEnvironmentModel.Type)

  def withEnvironmentId(value: String) = set(Id, value)
  def withName(value: String)          = set(Name, value)
  def withType(value: String)          = set(InstanceEnvironmentModel.Type, value)

  override def componentId: String = environmentId().option().getOrElse("environment")
}

object InstanceEnvironment {
  def apply(): InstanceEnvironment = InstanceEnvironment(Fields(), Annotations())
}

object InstanceEnvironmentModel extends ModelDefaultBuilder {

  val Id = Field(
      Str,
      Anypoint + "environmentID"
  )

  val Name = Field(
      Str,
      Core + "name"
  )

  val Type = Field(
      Str,
      Anypoint + "environmentType"
  )

  override def modelInstance: AmfObject = InstanceEnvironment()

  override def fields: List[Field] = List(Id, Name, Type)

  override val `type`: scala.List[ValueType] = List(ValueType(Management, "Environment"))
}
