package org.mulesoft.apb.project.client.scala.model

import amf.core.client.scala.model.document.{BaseUnit, EncodesModel}
import amf.core.client.scala.validation.AMFValidationResult
import amf.core.internal.adoption.IdAdopter
import amf.core.internal.metamodel.Field
import amf.core.internal.metamodel.Type.Str
import amf.core.internal.metamodel.document.BaseUnitModel
import amf.core.internal.metamodel.document.DocumentModel.Encodes
import amf.core.internal.parser.domain.{Annotations, Fields}
import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import org.mulesoft.apb.project.client.scala.extensions.APIProjectExtension
import org.mulesoft.apb.project.internal.BuildResult
import org.mulesoft.apb.project.internal.descriptor.ApiProjectNamespaces.Management
import org.mulesoft.apb.project.internal.metamodel.ProjectDocumentModel
import org.mulesoft.apb.project.internal.metamodel.ProjectInfoModel.{Contract, Instances}
import org.mulesoft.apb.project.internal.view.ProjectView

class ProjectDocument(override val fields: Fields, override val annotations: Annotations)
    extends BaseUnit
    with EncodesModel {

  /** Meta data for the document */
  override def meta: BaseUnitModel = ProjectDocumentModel

  /** Returns the list document URIs referenced from the document that has been parsed to generate this model */
  override def references: Seq[BaseUnit] = Nil

  /** Encoded DomainElement described in the document element. */
  override def encodes: ProjectInfo = fields(Encodes)

  /** Value , path + field value that is used to compose the id when the object its adopted */
  override def componentId: String = "project"
}

object ProjectDocument {
  def apply() = new ProjectDocument(Fields(), Annotations())
}

case class Project(private[apb] val document: ProjectDocument) {

  private[apb] def projectInfo       = document.encodes
  def apiContract(): BaseUnit        = projectInfo.apiContract()
  def instances(): Seq[JsonLDObject] = projectInfo.instances()

  // TODO native-jsonld : add properties of the descriptor
}

object ProjectDocumentBuilder {
  val INSTANCE_ID_IRI: Field = Field(Str, Management + "instanceID")
}

case class ProjectDocumentBuilder(descriptor: ProjectDescriptor) {
  private val projectInfo                          = ProjectInfo()
  private val projectDocument                      = ProjectDocument().set(Encodes, projectInfo)
  private var extensions: Seq[APIProjectExtension] = List.empty
  private var contract: Option[BaseUnit]           = None
  private var errors: Seq[AMFValidationResult]     = Seq.empty
  private var instances: Seq[JsonLDObject]         = Seq.empty
  private var base: Option[String]                 = None

  def withContract(unit: BaseUnit, errors: Seq[AMFValidationResult]): ProjectDocumentBuilder = {
    contract = Some(unit)
    this.errors = this.errors ++ errors
    this
  }

  def withInstances(instances: List[JsonLDObject], errors: Seq[AMFValidationResult]): ProjectDocumentBuilder = {
    this.instances = this.instances ++ instances
    this.errors = this.errors ++ errors
    this
  }

  def withExtensions(extensions: List[APIProjectExtension]) = {
    this.extensions = extensions
    this
  }

  def withBase(base: String): ProjectDocumentBuilder = {
    this.base = Some(base)
    this
  }
  def build(): BuildResult[Project] = {
    addFieldsFromDescriptor(projectInfo, descriptor)
    val project = Project(projectDocument)
    adoptProject()
    // We must honor ids because they might come from cache. This means that ID generation of each part will be isolated to each.
    contract.foreach(projectInfo.setWithoutId(Contract, _))
    projectInfo.setArrayWithoutId(Instances, instances)
    extensions.foreach(extension => extension.extend(project))
    BuildResult(project, project.projectInfo.id, this.errors)
  }

  private def adoptProject(): Unit = {
    val idAdopter = buildIdAdopter()
    idAdopter.adoptFromRoot()
  }

  private def buildIdAdopter() = {
    val finalBase = this.base.getOrElse("http://project.aml#")
    new IdAdopter(projectDocument, finalBase)
  }

  private def addFieldsFromDescriptor(projectInfo: ProjectInfo, descriptor: ProjectDescriptor) = {
    ProjectView(projectInfo).view(descriptor.wrapped)
  }
}
