package org.mulesoft.apb.project.internal.dependency

import amf.core.client.scala.AMFGraphConfiguration
import amf.core.client.scala.resource.ResourceLoader
import amf.core.internal.unsafe.PlatformSecrets
import org.mulesoft.apb.project.client.scala.DependencySetResult
import org.mulesoft.apb.project.client.scala.dependency._
import org.mulesoft.apb.project.client.scala.descriptor.DescriptorParseResult
import org.mulesoft.apb.project.client.scala.environment.DependencyFetcher
import org.mulesoft.apb.project.client.scala.model.descriptor.ProjectDescriptor
import org.mulesoft.apb.project.client.scala.model.report.APBResult
import org.mulesoft.apb.project.internal.engine.{AnypointTreeBuilder, LegacyTreeBuilder}
import org.mulesoft.apb.project.internal.generated.DescriptorSchemaLoader.LegacyVersion
import org.mulesoft.apb.project.internal.parser.APBEnv

import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.Future

object DependencySetParser {

  def apply(dependencyFetcher: DependencyFetcher, loaders: List[ResourceLoader]): DependencySetParser = {
    new DependencySetParser(dependencyFetcher, loaders)
  }
}

class DependencySetParser(
    dependencyFetcher: DependencyFetcher,
    loaders: List[ResourceLoader],
    cacheBuilder: UnitCacheBuilder = CloneOnceUnitCacheBuilder
) extends PlatformSecrets {

  def build(project: ProjectDescriptor, errors: List[APBResult] = Nil): Future[DependencySetResult] = {
    val config = project.descriptorVersion match {
      case LegacyVersion => LegacyTreeBuilder(dependencyFetcher, loaders).build(project)
      case _             => AnypointTreeBuilder(dependencyFetcher, loaders, cacheBuilder).build(project)
    }
    config.map(c => DependencySetResult(c, errors ++ c.results))
  }

  def build(project: DescriptorParseResult): Future[DependencySetResult] = {
    val DescriptorParseResult(descriptor, results) = project
    build(descriptor, adaptErrorsLocation(results.toList))
  }

  def buildFromContent(descriptorAsString: String): Future[DependencySetResult] = {
    val descriptor = APBEnv.getHandler().parse(descriptorAsString)
    build(descriptor)
  }

  def buildFromDirectory(directory: String): Future[DependencySetResult] = {
    val eventualContent = fetchDescriptorFile(directory)
    eventualContent.flatMap(c => buildFromContent(c.stream.toString))
  }

  private def adaptErrorsLocation(errors: List[APBResult]): List[APBResult] = {
    errors.map(er => APBResult.forAspect(er.copy(location = Some("exchange.json")), er.aspect))
  }

  private def fetchDescriptorFile(directory: String) = {
    val config = AMFGraphConfiguration.empty().withResourceLoaders(loaders)
    platform.fetchContent(directory + "/" + APBEnv.descriptorFileName, config)
  }
}
