package org.mulesoft.apb.project.client.scala.model.management

import amf.shapes.client.scala.model.domain.jsonldinstance.JsonLDObject
import amf.shapes.internal.domain.metamodel.jsonldschema.JsonLDEntityModel
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.apb.project.client.scala.instances.APIInstanceModel.{POLICY_BINDING_TERM, SERVICE_TERM}
import org.mulesoft.apb.project.client.scala.model.DynamicObject
import org.mulesoft.apb.project.client.scala.model.management.SchemaIris.{
  CONFIG,
  CORE,
  DESTINATION_REF,
  ENVIRONMENT,
  METADATA,
  NAME,
  NAMESPACE,
  ORDER,
  POLICIES,
  POLICY_BINDING_RULES,
  POLICY_REF,
  SERVICE,
  SPEC,
  TARGET_REF
}

import scala.reflect.ClassTag

/** PolicyBinding, either inlined or standalone resource. TODO: implement getters and setters for standalone mode
  * @param init
  *   initial value
  * @param inlined
  *   indicates if this is an inlined PolicyBindings. InlinedBindings do not have a 'SPEC' object, they are directly the
  *   'SPEC' object
  */
case class PolicyBinding(override private[apb] val internal: JsonLDObject, inlined: Boolean)
    extends DynamicObject
    with HasTargetRef
    with HasAnnotations
    with HasName {
  def service: Service = {
    val service = spec.getObject(CONFIG).getObject(SERVICE)
    Service(service)
  }

  def serviceIfPresent: Option[Service] = {
    spec.getObjectIfPresent(CONFIG).flatMap(_.getObjectIfPresent(SERVICE)).map(Service(_))
  }

  def removeService(): PolicyBinding = {
    getObjectIfPresent(CONFIG).map { spec =>
      spec.graph.removeField(SERVICE)
//      spec.removeProperty(SERVICE)
    }
    this
  }

  def withService(name: String, address: String): Service = {
    val service = Service(name, address)
    withService(service)
    service
  }

  def withService(service: Service): this.type = {
    update(SPEC) { spec =>
      spec.update(CONFIG) { config =>
        config.withProperty(SERVICE, service)
      }
    }
    this
  }

  def order: Int = spec.getScalar[Int](ORDER)

  def withOrder(order: Int): this.type = {
    update(SPEC) { spec =>
      spec.withProperty(ORDER, order)
    }
    this
  }

  def policyRef: PolicyRef = {
    val ref = spec.getObject(POLICY_REF)
    PolicyRef(ref.internal)
  }

  def withPolicyRef(name: String, namespace: Option[String] = None): this.type = {
    update(SPEC) { spec =>
      val init = PolicyRef(name)
      namespace.foreach(init.withNamespace)
      spec.withProperty(POLICY_REF, init)
    }
    this
  }

  private[apb] def getPolicyRefName: Option[String] =
    spec.getObjectIfPresent(POLICY_REF).flatMap(_.getScalarIfPresent[String](NAME))

  private[apb] def getDestinationRefName: Option[String] = spec
    .getObjectIfPresent(CONFIG)
    .flatMap(_.getObjectIfPresent(DESTINATION_REF))
    .flatMap(_.getScalarIfPresent[String](NAME))

  def getScalarConfig[T](name: String)(implicit ct: ClassTag[T]): T = {
    val config = spec.getObject(CONFIG)
    config.getScalar[T](s"$CORE#$name")
  }

  def getObjectConfig(name: String): DynamicObject = {
    val config = spec.getObject(CONFIG)
    config.getObject(s"$CORE#$name")
  }

  private def updateConfig(fn: (DynamicObject => DynamicObject)): this.type = {
    update(SPEC) { spec =>
      spec.update(CONFIG) { config =>
        fn(config)
      }
    }
    this
  }
  def withConfig(name: String, value: String): this.type = {
    updateConfig((config: DynamicObject) => config.withProperty(s"$CORE#$name", value))
  }

  def withConfig(name: String, value: Int): this.type = {
    updateConfig((config: DynamicObject) => config.withProperty(s"$CORE#$name", value))
  }

  def withConfig(name: String, value: Boolean): this.type = {
    updateConfig((config: DynamicObject) => config.withProperty(s"$CORE#$name", value))
  }

  def withConfig(name: String, value: Float): this.type = {
    updateConfig((config: DynamicObject) => config.withProperty(s"$CORE#$name", value))
  }

  def withConfig(name: String, value: Double): this.type = {
    updateConfig((config: DynamicObject) => config.withProperty(s"$CORE#$name", value))
  }

  def withConfig(name: String, value: DynamicObject): this.type = {
    updateConfig((config: DynamicObject) => config.withProperty(s"$CORE#$name", value))
  }

  def rules: Seq[PolicyBindingRule] = {
    val rules = spec.getObjectArray(POLICY_BINDING_RULES)
    rules.zipWithIndex
      .map { case (ruleDef, index) =>
        PolicyBindingRule(ruleDef)(index)
      }
  }

  def withRule(path: String, method: String, host: String): PolicyBindingRule = {
    val startingRules = rules
    val init          = PolicyBindingRule(path, method, host)(startingRules.length)
    update(SPEC) { spec =>
      spec.withProperty(POLICY_BINDING_RULES, startingRules :+ init)
    }
    init
  }
}

object PolicyBinding {
  def apply(name: String): PolicyBinding = apply(name, true)

  def apply(internal: JsonLDObject): PolicyBinding = new PolicyBinding(internal, false)

  def apply(name: String, inlined: Boolean): PolicyBinding = {
    val path  = JsonPath.empty
    val model = JsonLDEntityModel(List(POLICY_BINDING_TERM), Nil, path)
    val init  = JsonLDObject.empty(model, path)
    PolicyBinding(init, inlined)
      .withProperty("http://a.ml/vocabularies/core#apiVersion", Constants.apiVersion)
      .withProperty("http://a.ml/vocabularies/core#kind", Constants.policyBindingKind)
      .withName(name)
  }
}
