package org.mulesoft.apb.project.internal.engine

import amf.core.client.common.validation.SeverityLevels
import org.mulesoft.apb.project.client.scala.dependency.{Dependency, ParsedDependency, ProvidedDependency}
import org.mulesoft.apb.project.client.scala.model.descriptor.{DependencyScope, ExtensionScope, Gav, ProjectDependency}
import org.mulesoft.apb.project.client.scala.model.report.{APBResult, ProjectAspects}
import org.mulesoft.apb.project.internal.validations.ProjectValidations.SuggestedExtensionDependency

object SuggestedExtensionAdditions {

  def apply(project: Gav, dependencies: Seq[ParsedDependency]): Seq[APBResult] = {
    val notPresentExtensionsByProfile = buildSuggestedExtensionByProfile(dependencies)
    createErrorResults(project, notPresentExtensionsByProfile)
  }

  private def createErrorResults(project: Gav, notPresentExtensionsByProfile: Map[Gav, Set[(Gav, DependencyScope)]]) = {
    notPresentExtensionsByProfile.flatMap { case (profileGav, extensionGavs) =>
      extensionGavs.map(extensionGav => suggestion(project, profileGav, extensionGav._1, extensionGav._2))
    }.toList
  }

  private def buildSuggestedExtensionByProfile(
      currentDependencies: Seq[ParsedDependency]
  ): Map[Gav, Set[(Gav, DependencyScope)]] = {
    val projectExtensionGavs        = currentDependencies.map(d => (d.descriptor.gav(), d.scope())).toSet
    val extensionsUsedByEachProfile = currentDependencies.map(p => p.descriptor.gav() -> getProvidedExtensions(p)).toMap
    val notPresentExtensionsByProfile = extensionsUsedByEachProfile.mapValues(x => x.diff(projectExtensionGavs))
    notPresentExtensionsByProfile
  }

  private def suggestedExtensionMessage(profileGav: Gav, extensionGav: Gav, scope: DependencyScope) = {
    s"${scope.scope} dependency: ${extensionGav.path()} should be added when using validation dependency: ${profileGav.path()}"
  }

  private def filterExtensionDependencies(dependency: Dependency): Seq[ProjectDependency] = {
    dependency.descriptor.dependencies().filter(p => p.scope == ExtensionScope)
  }

  private def getProvidedExtensions(dependency: Dependency): Set[(Gav, DependencyScope)] =
    filterExtensionDependencies(dependency).filter(x => x.shouldBeProvided).map(d => (d.gav, d.scope)).toSet

  private def suggestion(project: Gav, dependency: Gav, provided: Gav, scope: DependencyScope): APBResult = {
    APBResult(
        suggestedExtensionMessage(dependency, provided, scope),
        SeverityLevels.WARNING,
        "",
        None,
        SuggestedExtensionDependency.id,
        None,
        Some(project.path()),
        ProvidedDependency(dependency, provided, scope),
        ProjectAspects.TREE
    )
  }
}
